/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.tasks

import android.app.AlertDialog
import android.content.Context
import android.content.DialogInterface
import com.moengage.core.LogLevel
import com.moengage.core.internal.global.GlobalResources
import com.moengage.core.internal.model.ResultFailure
import com.moengage.core.internal.model.ResultSuccess
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.accountMetaForInstance
import com.moengage.core.internal.utils.getDeviceType
import com.moengage.inapp.internal.ERROR_MESSAGE_COULD_NOT_CREATE_VIEW
import com.moengage.inapp.internal.ERROR_MESSAGE_INAPP_EXCEEDING_DEVICE_DIMENSION
import com.moengage.inapp.internal.ERROR_MESSAGE_TEST_CAMPAIGN
import com.moengage.inapp.internal.ERROR_MESSAGE_UNSUPPORTED_ORIENTATION
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.InAppInstanceProvider
import com.moengage.inapp.internal.InAppModuleManager
import com.moengage.inapp.internal.canShowInAppInCurrentOrientation
import com.moengage.inapp.internal.getCurrentOrientation
import com.moengage.inapp.internal.getViewCreationMeta
import com.moengage.inapp.internal.isInAppExceedingScreen
import com.moengage.inapp.internal.isModuleEnabled
import com.moengage.inapp.internal.model.CampaignPayload
import com.moengage.inapp.internal.model.NativeCampaignPayload
import com.moengage.inapp.internal.repository.InAppFileManager
import com.moengage.inapp.model.CampaignData
import com.moengage.inapp.model.SelfHandledCampaign
import com.moengage.inapp.model.SelfHandledCampaignData

/**
 * @author Arshiya Khanum
 */
internal class ShowTestInApp(
    private val context: Context,
    private val sdkInstance: SdkInstance,
    private val campaignId: String
) {

    private val tag = "${InAppConstants.MODULE_TAG}ShowTestInApp"

    internal fun show() {
        try {
            val repository = InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance)
            sdkInstance.logger.log { "$tag show() : processing test in-app" }
            if (!isModuleEnabled(context, sdkInstance)) return
            if (campaignId.isBlank()) {
                sdkInstance.logger.log { "$tag show() : Empty campaign id. Cannot show test in-app." }
                return
            }
            InAppFileManager(
                context,
                sdkInstance
            ).deleteHtmlAssetsForCampaignIds(setOf(campaignId))

            val campaignResponse =
                repository.fetchTestCampaignPayload(campaignId, getDeviceType(context))
            if (campaignResponse == null) {
                showErrorDialog("$ERROR_MESSAGE_TEST_CAMPAIGN Draft-id: $campaignId")
                return
            }
            when (campaignResponse) {
                is ResultFailure<*> -> {
                    val errorMessage = campaignResponse.data as String
                    showErrorDialog("$errorMessage Draft-Id: $campaignId")
                }
                is ResultSuccess<*> -> {
                    displayTestInAppIfPossible(campaignResponse.data as CampaignPayload)
                }
            }
            sdkInstance.logger.log { "$tag show() : Completed showing test-inapp" }
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag show() : " }
        }
    }

    private fun displayTestInAppIfPossible(payload: CampaignPayload) {
        val controller = InAppInstanceProvider.getControllerForInstance(sdkInstance)
        if (InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED == payload.templateType) {
            payload as NativeCampaignPayload
            val listener = InAppInstanceProvider.getCacheForInstance(sdkInstance)
                .selfHandledListener ?: return
            val campaignData = payload.customPayload ?: return
            val data = SelfHandledCampaignData(
                CampaignData(payload.campaignId, payload.campaignName, payload.campaignContext),
                accountMetaForInstance(sdkInstance),
                SelfHandledCampaign(campaignData, payload.dismissInterval)
            )
            GlobalResources.mainThread.post {
                listener.onSelfHandledAvailable(data)
            }
            return
        }
        val meta = getViewCreationMeta(context)
        val view = controller.viewHandler.buildInApp(payload, meta)
        if (view == null) {
            sdkInstance.logger.log { "$tag displayTestInAppIfPossible() : Cannot show in-app. View creation failed." }
            showErrorDialog("$ERROR_MESSAGE_COULD_NOT_CREATE_VIEW Draft-id: $campaignId")
            return
        }
        if (isInAppExceedingScreen(context, view)) {
            showErrorDialog(ERROR_MESSAGE_INAPP_EXCEEDING_DEVICE_DIMENSION)
            return
        }
        if (!canShowInAppInCurrentOrientation(
                getCurrentOrientation(context),
                payload.supportedOrientations
            )
        ) {
            sdkInstance.logger.log { "$tag displayTestInAppIfPossible() : Cannot show in-app in the current orientation." }
            showErrorDialog(ERROR_MESSAGE_UNSUPPORTED_ORIENTATION)
            return
        }
        val activity = InAppModuleManager.getActivity() ?: return
        controller.viewHandler.addInAppToViewHierarchy(activity, view, payload)
    }

    private fun showErrorDialog(errorMessage: String) {
        val activity = InAppModuleManager.getActivity() ?: return
        val alertDialog = AlertDialog.Builder(activity)
        alertDialog.setMessage(errorMessage).setTitle("Test in-app error")
            .setPositiveButton("OK") { dialog: DialogInterface, _: Int -> dialog.dismiss() }

        activity.runOnUiThread {
            val dialog = alertDialog.create()
            dialog.show()
        }
    }
}