/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inapp.internal.tasks

import android.content.Context
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.currentSeconds
import com.moengage.inapp.internal.InAppConstants
import com.moengage.inapp.internal.InAppInstanceProvider
import com.moengage.inapp.internal.model.enums.StateUpdateType
import com.moengage.inapp.internal.model.meta.CampaignState
import com.moengage.inapp.internal.repository.PayloadMapper

/**
 * @author Arshiya Khanum
 */
internal class UpdateCampaignState(
    private val context: Context,
    private val sdkInstance: SdkInstance,
    private val updateType: StateUpdateType,
    private val campaignId: String,
    private val isSelfHandled: Boolean
) {

    private val tag = "${InAppConstants.MODULE_TAG}UpdateCampaignState"

    internal fun update() {
        try {
            sdkInstance.logger.log { "$tag update() : Update State: $updateType, Campaign-id:$campaignId" }
            val currentTime = currentSeconds()
            val repository = InAppInstanceProvider.getRepositoryForInstance(context, sdkInstance)
            val campaignEntity = repository.getCampaignById(campaignId) ?: return
            val campaign = PayloadMapper().campaignEntityToCampaign(campaignEntity)
            if (isSelfHandled && campaign.campaignMeta.templateType != InAppConstants.IN_APP_TEMPLATE_TYPE_SELF_HANDLED
            ) {
                sdkInstance.logger.log { "$tag update() : Expected template type was self-handled. Not a self handled campaign will ignore update." }
                return
            }
            repository.updateLastShowTime(currentTime)
            val updatedState = CampaignState(
                campaign.campaignState.showCount + 1,
                currentTime,
                campaign.campaignState.isClicked
            )
            val updateCount =
                repository.updateCampaignState(updatedState, campaign.campaignMeta.campaignId)
            repository.updateCache()
            sdkInstance.logger.log { "$tag update() : State Updates: $campaignId, Count: $updateCount" }
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag update() : " }
        }
    }
}