/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.inapp.internal.widgets.ratingbar

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapShader
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Path
import android.graphics.RectF
import android.graphics.Shader
import android.util.AttributeSet
import android.widget.RatingBar
import androidx.annotation.Keep
import com.moengage.inapp.R
import com.moengage.inapp.internal.model.enums.RatingType
import kotlin.math.cos
import kotlin.math.min
import kotlin.math.sin

/**
 * Base class for Custom RatingBar
 * @since 7.0.0
 */
@Suppress("SameParameterValue")
@SuppressLint("AppCompatCustomView")
@Keep
public abstract class BaseRatingBar : RatingBar {
    private var defaultColorOn = Color.rgb(0x61, 0x61, 0x61)
    private val defaultColorOff = Color.TRANSPARENT
    private val polygonVertices = 5
    private var strokeWidth = -1 // width of the outline
    private val paintInside = Paint()
    private val paintOutline = Paint()
    private var path = Path()
    private val rectangle = RectF()
    private val interiorAngleModifier = 2.2f
    private val dp = resources.displayMetrics.density
    private var starSize = 0f
    private lateinit var colorsJoined: Bitmap
    private val ratingType: RatingType

    @JvmOverloads
    public constructor(
        context: Context,
        ratingType: RatingType,
        attrs: AttributeSet? = null
    ) : super(
        context,
        attrs
    ) {
        this.ratingType = ratingType
        getXmlAttrs(context, attrs)
        init()
    }

    public constructor(
        context: Context,
        ratingType: RatingType,
        attrs: AttributeSet?,
        defStyleAttr: Int
    ) : super(context, attrs, defStyleAttr) {
        this.ratingType = ratingType
        getXmlAttrs(context, attrs)
        init()
    }

    private fun init() {
        paintInside.isAntiAlias = true
        paintOutline.strokeWidth = strokeWidth.toFloat()
        paintOutline.style = Paint.Style.STROKE
        paintOutline.strokeJoin = Paint.Join.ROUND // Remove this line to create pointy stars
        paintOutline.isAntiAlias = true
    }

    @Synchronized
    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        val desiredWidth = (20 * dp * numStars).toInt()
        val widthMode = MeasureSpec.getMode(widthMeasureSpec)
        val widthSize = MeasureSpec.getSize(widthMeasureSpec)
        val heightMode = MeasureSpec.getMode(heightMeasureSpec)
        val heightSize = MeasureSpec.getSize(heightMeasureSpec)

        // Measure Width
        val width = when (widthMode) {
            MeasureSpec.EXACTLY -> {
                // Must be this size
                widthSize
            }

            MeasureSpec.AT_MOST -> {
                // Can't be bigger than...
                min(desiredWidth, widthSize)
            }

            else -> {
                // Be whatever you want
                desiredWidth
            }
        }

        // Measure Height
        val height = when (heightMode) {
            MeasureSpec.EXACTLY -> {
                // Must be this size
                heightSize
            }

            MeasureSpec.AT_MOST -> {
                // Can't be bigger than...
                min(heightSize, width / numStars)
            }

            else -> {
                // Be whatever you want
                width / numStars
            }
        }

        // If starSize matches getHeight, the tips of the star can get cut off due to strokeWidth being added to the polygon size.
        // Make it a bit smaller to avoid this. Also decrease star size and spread them out rather than cutting them off if the
        // height is insufficient for the width.
        starSize = min(height, width / numStars).toFloat()
        if (strokeWidth < 0) strokeWidth = (starSize / 15).toInt()
        starSize -= strokeWidth.toFloat()

        paintOutline.strokeWidth = strokeWidth.toFloat()

        // MUST CALL THIS
        setMeasuredDimension(width, height)
    }

    // Create a star polygon with any number of vertices, down to 2, which creates a diamond
    // If you enter 0 vertices, you get a circle
    private fun createStarBySize(size: Float, steps: Int): Path {
        // draw a simple circle if steps == 0
        if (steps == 0) {
            path.addOval(RectF(0f, 0f, size, size), Path.Direction.CW)
            path.close()
            return path
        }
        val halfSize = size / 2.0f
        val radius = halfSize / interiorAngleModifier // Adjusts "pointiness" of stars
        val degreesPerStep = Math.toRadians((360.0f / steps.toFloat()).toDouble()).toFloat()
        val halfDegreesPerStep = degreesPerStep / 2.0f
        path.fillType = Path.FillType.EVEN_ODD
        val max = (2.0f * Math.PI).toFloat()
        path.moveTo(halfSize, 0f)
        var step = 0.0
        while (step < max) {
            path.lineTo(
                (halfSize - halfSize * sin(step)).toFloat(),
                (halfSize - halfSize * cos(step)).toFloat()
            )
            path.lineTo(
                (halfSize - radius * sin(step + halfDegreesPerStep)).toFloat(),
                (halfSize - radius * cos(step + halfDegreesPerStep)).toFloat()
            )
            step += degreesPerStep.toDouble()
        }
        path.close()
        return path
    }

    override fun onDraw(canvas: Canvas) {
        // Default RatingBar changes color when pressed. This replicates the effect
        path.rewind()
        path = createStarBySize(starSize, polygonVertices)
        for (i in 0 until numStars) {
            val position = if (rating > 0) rating.toInt() else i + 1
            // Using the color value in `background`, since the border and background color will
            // be same in star rating icon
            val colorOn = getColorOn(position) ?: defaultColorOn
            val colorOff = getColorOff(position) ?: defaultColorOff
            paintInside.shader = updateShader(colorOn, colorOff)
            path.computeBounds(rectangle, true)
            path.offset(
                (i + .5f) * width / numStars - rectangle.centerX(),
                height / 2 - rectangle.centerY()
            )
            // Default RatingBar only shows fractions in the interior, not the outline.
            paintOutline.color = colorOn
            canvas.drawPath(path, paintInside)
            canvas.drawPath(path, paintOutline)
        }
    }

    // Create a BitmapShader, which is used to show fractions of stars
    private fun updateShader(colorOn: Int, colorOff: Int): BitmapShader {
        // Bitmap of width 0 will cause a crash. Make sure it's a positive number.
        val ratingWidth = (rating * width / numStars).toInt()
        if (ratingWidth <= 0 || width - ratingWidth <= 0) {
            colorsJoined = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888)
            colorsJoined.eraseColor(if (ratingWidth <= 0) colorOff else colorOn)
        } else {
            val colorLeft = Bitmap.createBitmap(ratingWidth, height, Bitmap.Config.ARGB_8888)
            colorLeft.eraseColor(colorOn)
            val colorRight =
                Bitmap.createBitmap(width - ratingWidth, height, Bitmap.Config.ARGB_8888)
            colorRight.eraseColor(colorOff)
            colorsJoined = combineBitmaps(colorLeft, colorRight)
        }

        return BitmapShader(colorsJoined, Shader.TileMode.CLAMP, Shader.TileMode.CLAMP)
    }

    // Combine two bitmaps side by side for use as a BitmapShader
    private fun combineBitmaps(leftBitmap: Bitmap, rightBitmap: Bitmap): Bitmap {
        colorsJoined = Bitmap.createBitmap(
            leftBitmap.width + rightBitmap.width,
            leftBitmap.height,
            Bitmap.Config.ARGB_8888
        )
        val comboImage = Canvas(colorsJoined)
        comboImage.drawBitmap(leftBitmap, 0f, 0f, null)
        comboImage.drawBitmap(rightBitmap, leftBitmap.width.toFloat(), 0f, null)
        return colorsJoined
    }

    // Set any XML attributes that may have been specified
    private fun getXmlAttrs(context: Context, attrs: AttributeSet?) {
        val a = context.theme.obtainStyledAttributes(attrs, R.styleable.MoERatingBar, 0, 0)
        defaultColorOn = try {
            a.getInteger(
                R.styleable.MoERatingBar_starColor,
                Color.rgb(0x61, 0x61, 0x61)
            )
        } finally {
            a.recycle()
        }
    }

    public abstract fun getColorOn(position: Int): Int?

    public abstract fun getColorOff(position: Int): Int?
}