/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inbox.core

import android.content.Context
import androidx.annotation.Nullable
import androidx.annotation.WorkerThread
import com.moengage.core.LogLevel
import com.moengage.core.PUSH_NOTIFICATION_COUPON_CODE
import com.moengage.core.internal.SdkInstanceManager
import com.moengage.core.internal.global.GlobalResources
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.accountMetaForInstance
import com.moengage.inbox.core.internal.InboxCoreInstanceProvider
import com.moengage.inbox.core.internal.InboxProcessor
import com.moengage.inbox.core.internal.MODULE_TAG
import com.moengage.inbox.core.listener.OnMessagesAvailableListener
import com.moengage.inbox.core.listener.UnClickedCountListener
import com.moengage.inbox.core.model.InboxData
import com.moengage.inbox.core.model.InboxMessage
import com.moengage.inbox.core.model.UnClickedCountData

/**
 * Helper class for Inbox-core Related features.
 *
 * @author Umang Chamaria
 * @since 1.0.00
 */
public class MoEInboxHelper private constructor() {

    private val tag = "${MODULE_TAG}MoEInboxHelper"

    /**
     * Marks the given message as clicked and tracks a click event for the same. This API updates
     * the message for the account configured as default.
     *
     * @param context instance of [Context]
     * @param inboxMessage instance of [InboxMessage]
     * @since 1.0.00
     */
    public fun trackMessageClicked(context: Context, inboxMessage: InboxMessage) {
        val instance = SdkInstanceManager.defaultInstance ?: return
        trackMessageClicked(context, inboxMessage, instance)
    }

    /**
     * Marks the given message as clicked and tracks a click event for the same. This API updates
     * the message for the given account.
     *
     * @param context instance of [Context]
     * @param inboxMessage instance of [InboxMessage]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 2.0.0
     */
    public fun trackMessageClicked(context: Context, inboxMessage: InboxMessage, appId: String) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        trackMessageClicked(context, inboxMessage, instance)
    }

    private fun trackMessageClicked(
        context: Context,
        inboxMessage: InboxMessage,
        sdkInstance: SdkInstance
    ) {
        try {
            sdkInstance.taskHandler.submitRunnable {
                InboxProcessor().trackMessageClicked(context, sdkInstance, inboxMessage)
            }
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag trackMessageClicked() : " }
        }
    }

    /**
     * Deletes the given message from inbox. This API deletes the message for the account
     * configured as default.
     *
     * @param context instance of [Context]
     * @param inboxMessage instance of [InboxMessage]
     * @since 1.0.00
     */
    public fun deleteMessage(context: Context, inboxMessage: InboxMessage) {
        val instance = SdkInstanceManager.defaultInstance ?: return
        deleteMessage(context, inboxMessage, instance)
    }

    /**
     * Deletes the given message from inbox. This API deletes the message for the given account.
     *
     * @param context instance of [Context]
     * @param inboxMessage instance of [InboxMessage]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 2.0.0
     */
    public fun deleteMessage(context: Context, inboxMessage: InboxMessage, appId: String) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return
        deleteMessage(context, inboxMessage, instance)
    }

    private fun deleteMessage(
        context: Context,
        inboxMessage: InboxMessage,
        sdkInstance: SdkInstance
    ) {
        try {
            if (inboxMessage.id == -1L) return
            sdkInstance.taskHandler.submitRunnable {
                InboxCoreInstanceProvider.getRepositoryForInstance(context, sdkInstance)
                    .deleteMessage(inboxMessage)
            }
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag deleteMessage() : " }
        }
    }

    /**
     * Gets all the messages saved in the inbox in the listener. This API fetches the message for
     * the account configured as default.
     *
     * @param context instance of [Context]
     * @param listener instance of [OnMessagesAvailableListener]
     * @since 1.0.00
     */
    public fun fetchAllMessagesAsync(context: Context, listener: OnMessagesAvailableListener) {
        when (val instance = SdkInstanceManager.defaultInstance) {
            is SdkInstance -> fetchAllMessagesAsync(context, listener, instance)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag fetchAllMessagesAsync(): Default instance not initialised." }
                GlobalResources.mainThread.post {
                    listener.onMessagesAvailable(null)
                }
            }
        }
    }

    /**
     * Gets all the messages saved in the inbox in the listener. This API fetches the message for
     * the given account.
     *
     * @param context instance of [Context]
     * @param listener instance of [OnMessagesAvailableListener]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 2.0.0
     */
    public fun fetchAllMessagesAsync(
        context: Context,
        appId: String,
        listener: OnMessagesAvailableListener
    ) {
        when (val instance = SdkInstanceManager.getInstanceForAppId(appId)) {
            is SdkInstance -> fetchAllMessagesAsync(context, listener, instance)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag fetchAllMessagesAsync(): Instance not initialised: $appId" }
                GlobalResources.mainThread.post {
                    listener.onMessagesAvailable(null)
                }
            }
        }
    }

    private fun fetchAllMessagesAsync(
        context: Context,
        listener: OnMessagesAvailableListener,
        sdkInstance: SdkInstance
    ) {
        try {
            InboxProcessor().fetchMessagesAsync(context, sdkInstance, listener)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag fetchAllMessagesAsync() : " }
        }
    }

    /**
     * Gets all the messages saved in the inbox for the account configured as default.
     *
     * @param context instance of [Context]
     * @return instance of [InboxData] or null if the account is not configured
     * @since 1.0.00
     */
    @WorkerThread
    public fun fetchAllMessages(context: Context): InboxData? {
        return when (val instance = SdkInstanceManager.defaultInstance) {
            is SdkInstance -> fetchAllMessages(context, instance)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag fetchAllMessages(): Default instance not initialised." }
                null
            }
        }
    }

    /**
     * Gets all the messages saved in the inbox for the given account.
     *
     * @param context instance of [Context]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @return instance of [InboxData] or null if the account is not configured.
     * @since 2.0.0
     */
    @WorkerThread
    public fun fetchAllMessages(context: Context, appId: String): InboxData? {
        return when (val instance = SdkInstanceManager.getInstanceForAppId(appId)) {
            is SdkInstance -> fetchAllMessages(context, instance)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag fetchAllMessages(): Instance not initialised $appId" }
                null
            }
        }
    }

    private fun fetchAllMessages(context: Context, sdkInstance: SdkInstance): InboxData {
        return InboxProcessor().fetchMessages(context, sdkInstance)
    }

    /**
     * Returns the count of un-clicked messages in the inbox for the account configured as default.
     *
     * @param context Instance of [Context]
     * @return Instance of [UnClickedCountData] or null if the account is not configured
     * @since 1.0.00
     */
    @WorkerThread
    @Nullable
    public fun getUnClickedMessagesCount(context: Context): UnClickedCountData? {
        return when (val instance = SdkInstanceManager.defaultInstance) {
            is SdkInstance -> getUnClickedMessagesCount(context, instance)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag getUnClickedMessagesCount(): Default instance not initialised." }
                null
            }
        }
    }

    /**
     * Returns the count of un-clicked messages in the inbox for the given instance.
     *
     * @param context Instance of [Context]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @return Instance of [UnClickedCountData] or null if the account is not configured.
     * @since 2.0.0
     */
    @WorkerThread
    @Nullable
    public fun getUnClickedMessagesCount(context: Context, appId: String): UnClickedCountData? {
        return when (val instance = SdkInstanceManager.getInstanceForAppId(appId)) {
            is SdkInstance -> getUnClickedMessagesCount(context, instance)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag getUnClickedMessagesCount(): Instance not initialised $appId" }
                null
            }
        }
    }

    private fun getUnClickedMessagesCount(
        context: Context,
        sdkInstance: SdkInstance
    ): UnClickedCountData {
        try {
            return UnClickedCountData(
                accountMetaForInstance(sdkInstance),
                InboxCoreInstanceProvider.getRepositoryForInstance(context, sdkInstance)
                    .getUnClickedMessageCount()
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag getUnClickedMessagesCount() : " }
        }
        return UnClickedCountData(accountMetaForInstance(sdkInstance), 0)
    }

    /**
     * Returns the count of un-clicked messages in the inbox to the listener. This API fetches
     * the count for the account configured as default.
     *
     * @param context Instance of [Context]
     * @param listener Instance of [UnClickedCountListener]
     * @since 1.0.00
     */
    public fun getUnClickedMessagesCountAsync(context: Context, listener: UnClickedCountListener) {
        when (val instance = SdkInstanceManager.defaultInstance) {
            is SdkInstance -> getUnClickedMessagesCountAsync(context, listener, instance)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag getUnClickedMessagesCountAsync(): Default instance not initialised." }
                GlobalResources.mainThread.post {
                    listener.onCountAvailable(null)
                }
            }
        }
    }

    /**
     * Returns the count of un-clicked messages in the inbox to the listener. This API fetches
     * the count for the given account.
     *
     * @param context Instance of [Context]
     * @param listener Instance of [UnClickedCountListener]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 2.0.0
     */
    public fun getUnClickedMessagesCountAsync(
        context: Context,
        appId: String,
        listener: UnClickedCountListener
    ) {
        when (val instance = SdkInstanceManager.getInstanceForAppId(appId)) {
            is SdkInstance -> getUnClickedMessagesCountAsync(context, listener, instance)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag getUnClickedMessagesCountAsync() : Instance not initialised $appId" }
                GlobalResources.mainThread.post {
                    listener.onCountAvailable(null)
                }
            }
        }
    }

    private fun getUnClickedMessagesCountAsync(
        context: Context,
        listener: UnClickedCountListener,
        sdkInstance: SdkInstance
    ) {
        sdkInstance.taskHandler.submitRunnable {
            InboxProcessor().getUnClickedMessageCountAsync(context, sdkInstance, listener)
        }
    }

    /**
     * Checks whether the message has a coupon code associated to it.
     *
     * @param inboxMessage instance of [InboxMessage]
     * @return true if the message has a coupon code associated to it, else false.
     * @since 1.0.00
     */
    public fun hasCouponCode(inboxMessage: InboxMessage): Boolean {
        try {
            return inboxMessage.payload.has(PUSH_NOTIFICATION_COUPON_CODE)
        } catch (e: Exception) {
            Logger.print(LogLevel.ERROR, e) { "$tag hasCouponCode() : " }
        }
        return false
    }

    /**
     * Get the coupon code associated to the campaign, if any.
     *
     * @param inboxMessage instance of [InboxMessage]
     * @return Coupon code if exist, else empty string.
     * @since 1.0.00
     */
    public fun getCouponCode(inboxMessage: InboxMessage): String {
        try {
            return inboxMessage.payload.optString(PUSH_NOTIFICATION_COUPON_CODE, "")
        } catch (e: Exception) {
            Logger.print(LogLevel.ERROR, e) { "$tag getCouponCode() : " }
        }
        return ""
    }

    public companion object {
        private var instance: MoEInboxHelper? = null

        @JvmStatic
        public fun getInstance(): MoEInboxHelper {
            return instance ?: synchronized(MoEInboxHelper::class.java) {
                val inst = instance ?: MoEInboxHelper()
                instance = inst
                inst
            }
        }
    }

    /**
     * Gets all the messages filtered by message tag in the inbox for the instance configured as
     * default.
     *
     * @param context instance of [Context]
     * @param messageTag Inbox Message tag [InboxMessage]
     * @return Instance of [InboxData] or null if the instance is not configured.
     * @since 1.0.00
     */
    @WorkerThread
    @Nullable
    public fun fetchMessagesByTag(context: Context, messageTag: String): InboxData? {
        return when (val instance = SdkInstanceManager.defaultInstance) {
            is SdkInstance -> fetchMessagesByTag(context, messageTag, instance)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag fetchMessagesByTag() : Default instance not initialised." }
                null
            }
        }
    }

    /**
     * Gets all the messages filtered by message tag in the inbox for the given instance.
     *
     * @param context instance of [Context]
     * @param messageTag Inbox Message tag [InboxMessage]
     * @return instance of [InboxData] or null if the instance is not configured.
     * @since 2.0.0
     */
    @WorkerThread
    @Nullable
    public fun fetchMessagesByTag(context: Context, messageTag: String, appId: String): InboxData? {
        return when (val instance = SdkInstanceManager.getInstanceForAppId(appId)) {
            is SdkInstance -> fetchMessagesByTag(context, messageTag, instance)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag fetchMessagesByTag() : Instance not initialised $appId" }
                null
            }
        }
    }

    private fun fetchMessagesByTag(
        context: Context,
        messageTag: String,
        sdkInstance: SdkInstance
    ): InboxData {
        return InboxProcessor().fetchMessages(context, sdkInstance, messageTag)
    }

    /**
     * Return all the messages filtered by message tag in the inbox to the listener. This API
     * fetches messages for the account configured as default.
     *
     * @param context instance of [Context]
     * @param messageTag Tag for which messages should be filtered.
     * @param listener Instance of [OnMessagesAvailableListener]
     * @since 1.0.00
     */
    public fun fetchMessagesByTagAsync(
        context: Context,
        messageTag: String,
        listener: OnMessagesAvailableListener
    ) {
        when (val instance = SdkInstanceManager.defaultInstance) {
            is SdkInstance -> fetchMessageByTagAsync(context, messageTag, instance, listener)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag fetchMessagesByTagAsync() : Default instance not initialised." }
                GlobalResources.mainThread.post {
                    listener.onMessagesAvailable(null)
                }
            }
        }
    }

    /**
     * Return all the messages filtered by message tag in the inbox to the listener. This API
     * fetches messages for the account configured as default.
     *
     * @param context instance of [Context]
     * @param messageTag Tag for which messages should be filtered.
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @param listener Instance of [OnMessagesAvailableListener]
     * @since 2.0.0
     */
    public fun fetchMessagesByTagAsync(
        context: Context,
        messageTag: String,
        appId: String,
        listener: OnMessagesAvailableListener
    ) {
        when (val instance = SdkInstanceManager.getInstanceForAppId(appId)) {
            is SdkInstance -> fetchMessageByTagAsync(context, messageTag, instance, listener)
            else -> {
                Logger.print(LogLevel.ERROR) { "$tag fetchMessagesByTagAsync() : Instance not initialised $appId" }
                GlobalResources.mainThread.post {
                    listener.onMessagesAvailable(null)
                }
            }
        }
    }

    private fun fetchMessageByTagAsync(
        context: Context,
        messageTag: String,
        sdkInstance: SdkInstance,
        listener: OnMessagesAvailableListener
    ) {
        sdkInstance.taskHandler.submitRunnable {
            InboxProcessor().fetchMessagesAsync(context, sdkInstance, listener, messageTag)
        }
    }
}