/*
 * Copyright (c) 2014-2021 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inbox.core.internal

import android.content.Context
import com.moengage.core.LogLevel
import com.moengage.core.PUSH_NOTIFICATION_CAMPAIGN_ID
import com.moengage.core.Properties
import com.moengage.core.analytics.MoEAnalyticsHelper
import com.moengage.core.internal.EVENT_NOTIFICATION_CLICKED
import com.moengage.core.internal.executor.Job
import com.moengage.core.internal.global.GlobalResources
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.accountMetaForInstance
import com.moengage.inbox.core.listener.OnMessagesAvailableListener
import com.moengage.inbox.core.listener.UnClickedCountListener
import com.moengage.inbox.core.model.InboxData
import com.moengage.inbox.core.model.InboxMessage
import com.moengage.inbox.core.model.UnClickedCountData

/**
 * @author Umang Chamaria
 */
internal class InboxProcessor {

    private val tag = "${MODULE_TAG}InboxProcessor"

    fun trackMessageClicked(
        context: Context,
        sdkInstance: SdkInstance,
        inboxMessage: InboxMessage
    ) {
        try {
            if (inboxMessage.id == -1L) return
            InboxCoreInstanceProvider.getRepositoryForInstance(context, sdkInstance)
                .markMessageClicked(inboxMessage)
            val properties = Properties()
            properties.addAttribute(
                PUSH_NOTIFICATION_CAMPAIGN_ID,
                inboxMessage.campaignId
            )
            properties.addAttribute(SOURCE_ATTRIBUTE_NAME, SOURCE_ATTRIBUTE_VALUE)
            MoEAnalyticsHelper.trackEvent(
                context,
                EVENT_NOTIFICATION_CLICKED,
                properties,
                sdkInstance.instanceMeta.instanceId
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag trackMessageClicked() : " }
        }
    }

    fun fetchMessagesAsync(
        context: Context,
        sdkInstance: SdkInstance,
        listener: OnMessagesAvailableListener,
        messageTag: String = ""
    ) {
        sdkInstance.taskHandler.execute(
            Job(TAG_FETCH_MESSAGES, false) {
                try {
                    val inboxData = fetchMessages(context, sdkInstance, messageTag)
                    GlobalResources.mainThread.post {
                        listener.onMessagesAvailable(inboxData)
                    }
                } catch (e: Exception) {
                    sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag fetchMessagesAsync() : " }
                }
            }
        )
    }

    fun fetchMessages(context: Context, sdkInstance: SdkInstance, tag: String = ""): InboxData {
        try {
            return if (tag.isBlank()) {
                InboxData(
                    accountMetaForInstance(sdkInstance),
                    InboxCoreInstanceProvider.getRepositoryForInstance(context, sdkInstance)
                        .fetchAllMessages()
                )
            } else {
                InboxData(
                    accountMetaForInstance(sdkInstance),
                    InboxCoreInstanceProvider.getRepositoryForInstance(context, sdkInstance)
                        .fetchMessagesByTag(tag)
                )
            }
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag fetchMessages() : " }
        }
        return InboxData(accountMetaForInstance(sdkInstance), emptyList())
    }

    fun getUnClickedMessageCountAsync(
        context: Context,
        sdkInstance: SdkInstance,
        listener: UnClickedCountListener
    ) {
        try {
            val unClickedCount = getUnClickedMessageCount(context, sdkInstance)
            GlobalResources.mainThread.post {
                listener.onCountAvailable(
                    UnClickedCountData(
                        accountMetaForInstance(sdkInstance),
                        unClickedCount
                    )
                )
            }
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag getUnClickedMessageCountAsync() : " }
        }
    }

    fun getUnClickedMessageCount(context: Context, sdkInstance: SdkInstance): Long {
        return InboxCoreInstanceProvider.getRepositoryForInstance(context, sdkInstance)
            .getUnClickedMessageCount()
    }
}