/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inbox.core.internal.repository.local

import android.content.ContentValues
import android.content.Context
import android.database.Cursor
import com.moengage.core.LogLevel
import com.moengage.core.internal.CoreInternalHelper
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.model.database.DataAccessor
import com.moengage.core.internal.model.database.QueryParams
import com.moengage.core.internal.model.database.WhereClause
import com.moengage.core.internal.storage.database.contract.BASE_COLUMN_NAME_GTIME
import com.moengage.core.internal.storage.database.contract.BASE_COLUMN_NAME_ID
import com.moengage.core.internal.storage.database.contract.INBOX_COLUMN_NAME_CAMPAIGN_TAG
import com.moengage.core.internal.storage.database.contract.INBOX_COLUMN_NAME_IS_CAMPAIGN_CLICKED
import com.moengage.core.internal.storage.database.contract.PROJECTION_INBOX
import com.moengage.core.internal.storage.database.contract.TABLE_NAME_INBOX
import com.moengage.inbox.core.internal.MODULE_TAG
import com.moengage.inbox.core.model.InboxMessage

/**
 * @author Umang Chamaria
 * @since 1.0.00
 */

private const val DEFAULT_SORT_ORDER = "$BASE_COLUMN_NAME_GTIME DESC"

internal class LocalRepositoryImpl(
    private val context: Context,
    private val dataAccessor: DataAccessor,
    private val sdkInstance: SdkInstance
) : LocalRepository {

    private val tag = "${MODULE_TAG}LocalRepositoryImpl"

    private val marshallingHelper = MarshallingHelper(context, sdkInstance)

    override fun getUnClickedMessageCount(): Long {
        var count: Long = 0
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INBOX,
                QueryParams(
                    arrayOf(INBOX_COLUMN_NAME_IS_CAMPAIGN_CLICKED),
                    WhereClause(
                        "$INBOX_COLUMN_NAME_IS_CAMPAIGN_CLICKED = ? ",
                        arrayOf("0")
                    )
                )
            )
            if (cursor != null) count = cursor.count.toLong()
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag getUnClickedMessageCount() : " }
        } finally {
            cursor?.close()
        }
        return count
    }

    override fun fetchAllMessages(): List<InboxMessage> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INBOX,
                QueryParams(
                    projection = PROJECTION_INBOX,
                    whereClause = null,
                    orderBy = DEFAULT_SORT_ORDER
                )
            )
            if (cursor == null || !cursor.moveToFirst()) return emptyList()
            return marshallingHelper.cursorToMessages(cursor)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag fetchAllMessages() : " }
        } finally {
            cursor?.close()
        }
        return emptyList()
    }

    override fun deleteMessage(message: InboxMessage): Int {
        try {
            return dataAccessor.dbAdapter.delete(
                TABLE_NAME_INBOX,
                WhereClause(
                    "$BASE_COLUMN_NAME_ID = ? ",
                    arrayOf(message.id.toString())
                )
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag deleteMessage() : " }
        }
        return -1
    }

    override fun markMessageClicked(message: InboxMessage): Int {
        return markMessageClickedById(message.id)
    }

    override fun markMessageClickedById(id: Long): Int {
        var rowCount = -1
        try {
            val values = ContentValues()
            values.put(INBOX_COLUMN_NAME_IS_CAMPAIGN_CLICKED, 1)
            rowCount = dataAccessor.dbAdapter.update(
                TABLE_NAME_INBOX,
                values,
                WhereClause(
                    "$BASE_COLUMN_NAME_ID = ? ",
                    arrayOf(id.toString())
                )
            )
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag markMessageClickedById() : " }
        }
        return rowCount
    }

    override fun fetchMessagesByTag(msgTag: String): List<InboxMessage> {
        var cursor: Cursor? = null
        try {
            cursor = dataAccessor.dbAdapter.query(
                TABLE_NAME_INBOX,
                QueryParams(
                    projection = PROJECTION_INBOX,
                    whereClause = WhereClause(
                        "$INBOX_COLUMN_NAME_CAMPAIGN_TAG = ? ",
                        arrayOf(msgTag)
                    ),
                    orderBy = DEFAULT_SORT_ORDER
                )
            )
            if (cursor == null || !cursor.moveToFirst()) return emptyList()
            return marshallingHelper.cursorToMessages(cursor)
        } catch (e: Exception) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag fetchMessagesByTag() : " }
        } finally {
            cursor?.close()
        }
        return emptyList()
    }

    override fun storeInboxMessage(inboxMessage: InboxMessage): Long {
        return CoreInternalHelper.storePushCampaign(
            context,
            sdkInstance,
            marshallingHelper.inboxMessageToInboxData(inboxMessage)
        )
    }
}