/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.inbox.core.internal.repository.local

import android.content.Context
import android.database.Cursor
import com.moengage.core.LogLevel
import com.moengage.core.PUSH_NOTIFICATION_CAMPAIGN_ID
import com.moengage.core.PUSH_NOTIFICATION_IMAGE_URL
import com.moengage.core.PUSH_NOTIFICATION_MESSAGE
import com.moengage.core.PUSH_NOTIFICATION_NAVIGATION_ACTIVITY_NAME
import com.moengage.core.PUSH_NOTIFICATION_NAVIGATION_DEEPLINK
import com.moengage.core.PUSH_NOTIFICATION_NAVIGATION_DEEPLINK_LEGACY
import com.moengage.core.PUSH_NOTIFICATION_SUMMARY
import com.moengage.core.PUSH_NOTIFICATION_TITLE
import com.moengage.core.PUSH_NOTIFICATION_TYPE
import com.moengage.core.PUSH_NOTIFICATION_TYPE_DEEP_LINK_NOTIFICATION
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.model.database.entity.InboxEntity
import com.moengage.core.internal.storage.decryptValueIfRequired
import com.moengage.core.internal.utils.ISO8601Utils
import com.moengage.core.internal.utils.MoEUtils
import com.moengage.core.internal.utils.milliSecondsFromIsoString
import com.moengage.inbox.core.internal.MODULE_TAG
import com.moengage.inbox.core.model.InboxMessage
import com.moengage.inbox.core.model.MediaContent
import com.moengage.inbox.core.model.TextContent
import com.moengage.inbox.core.model.actions.Action
import com.moengage.inbox.core.model.actions.NavigationAction
import com.moengage.inbox.core.model.enums.ActionType
import com.moengage.inbox.core.model.enums.MediaType
import com.moengage.inbox.core.model.enums.NavigationType
import org.json.JSONObject
import java.util.*

/**
 * @author Umang Chamaria
 * Date: 2020/10/20
 */

private const val COLUMN_INDEX_ID: Int = 0
private const val COLUMN_INDEX_GTIME: Int = 1
private const val COLUMN_INDEX_MSG_DETAILS: Int = 2
private const val COLUMN_INDEX_MSG_CLICKED: Int = 3
private const val COLUMN_INDEX_MSG_TTL: Int = 4
private const val COLUMN_INDEX_MSG_TAG: Int = 5
private const val COLUMN_INDEX_MSG_CAMPAIGN_ID: Int = 6

internal class MarshallingHelper(
    private val context: Context,
    private val sdkInstance: SdkInstance
) {

    private val tag = "${MODULE_TAG}MarshallingHelper"

    fun cursorToMessages(cursor: Cursor): List<InboxMessage> {
        val inboxMessages = ArrayList<InboxMessage>(cursor.count)
        do {
            val message = cursorToMessage(cursor)
            if (message != null) inboxMessages.add(message)
        } while (cursor.moveToNext())
        return inboxMessages
    }

    fun inboxMessageToInboxData(inboxMessage: InboxMessage): InboxEntity {
        return InboxEntity(
            id = inboxMessage.id,
            campaignId = inboxMessage.campaignId,
            isClicked = if (inboxMessage.isClicked) 1 else 0,
            tag = inboxMessage.tag,
            receivedTime = milliSecondsFromIsoString(inboxMessage.receivedTime),
            expiry = milliSecondsFromIsoString(inboxMessage.expiry),
            payload = inboxMessage.payload.toString()
        )
    }

    internal fun cursorToMessage(cursor: Cursor): InboxMessage? {
        try {
            val messageJson = JSONObject(
                decryptValueIfRequired(
                    context,
                    sdkInstance,
                    cursor.getString(COLUMN_INDEX_MSG_DETAILS)
                )
            )
            var campaignId = cursor.getString(COLUMN_INDEX_MSG_CAMPAIGN_ID)
            if (campaignId.isNullOrBlank()) {
                campaignId = messageJson.getString(PUSH_NOTIFICATION_CAMPAIGN_ID)
            }
            return InboxMessage(
                id = cursor.getLong(COLUMN_INDEX_ID),
                campaignId = campaignId,
                textContent = TextContent(
                    title = messageJson.getString(PUSH_NOTIFICATION_TITLE),
                    message = messageJson.getString(PUSH_NOTIFICATION_MESSAGE),
                    summary = messageJson.optString(PUSH_NOTIFICATION_SUMMARY, "")
                ),
                action = actionFromPayload(messageJson),
                isClicked = cursor.getInt(COLUMN_INDEX_MSG_CLICKED) == 1,
                tag = cursor.getString(COLUMN_INDEX_MSG_TAG),
                receivedTime = ISO8601Utils.format(Date(cursor.getLong(COLUMN_INDEX_GTIME))),
                expiry = ISO8601Utils.format(Date(cursor.getLong(COLUMN_INDEX_MSG_TTL))),
                mediaContent = mediaFromPayload(messageJson),
                payload = messageJson
            )
        } catch (e: Exception) {
            Logger.print(LogLevel.ERROR, e) { "$tag cursorToMessage() : " }
        }
        return null
    }

    private fun actionFromPayload(payload: JSONObject): List<Action> {
        val notificationType = payload.getString(PUSH_NOTIFICATION_TYPE)
        val navigationType =
            if (PUSH_NOTIFICATION_TYPE_DEEP_LINK_NOTIFICATION == notificationType) {
                NavigationType.DEEP_LINK
            } else {
                if (payload.has(PUSH_NOTIFICATION_NAVIGATION_DEEPLINK_LEGACY)) {
                    NavigationType.RICH_LANDING
                } else {
                    NavigationType.SCREEN_NAME
                }
            }
        val url: String = when (navigationType) {
            NavigationType.DEEP_LINK -> {
                if (payload.has(PUSH_NOTIFICATION_NAVIGATION_DEEPLINK)) {
                    payload.getString(PUSH_NOTIFICATION_NAVIGATION_DEEPLINK)
                } else {
                    payload.getString(PUSH_NOTIFICATION_NAVIGATION_DEEPLINK_LEGACY)
                }
            }
            NavigationType.SCREEN_NAME -> {
                payload.getString(PUSH_NOTIFICATION_NAVIGATION_ACTIVITY_NAME)
            }
            else -> {
                payload.getString(PUSH_NOTIFICATION_NAVIGATION_DEEPLINK_LEGACY)
            }
        }
        val navigationAction =
            NavigationAction(
                ActionType.NAVIGATION,
                navigationType,
                url,
                MoEUtils.jsonToMap(payload)
            )
        return arrayListOf<Action>(navigationAction)
    }

    private fun mediaFromPayload(payload: JSONObject): MediaContent? {
        if (!payload.has(PUSH_NOTIFICATION_IMAGE_URL)) return null
        return MediaContent(
            MediaType.IMAGE,
            payload.getString(PUSH_NOTIFICATION_IMAGE_URL)
        )
    }

    internal fun cursorToInboxEntity(cursor: Cursor): InboxEntity? {
        try {
            val messageJson = JSONObject(
                decryptValueIfRequired(
                    context,
                    sdkInstance,
                    cursor.getString(COLUMN_INDEX_MSG_DETAILS)
                )
            )
            var campaignId = cursor.getString(COLUMN_INDEX_MSG_CAMPAIGN_ID)
            if (campaignId.isNullOrBlank()) {
                campaignId = messageJson.getString(PUSH_NOTIFICATION_CAMPAIGN_ID)
            }
            return InboxEntity(
                id = cursor.getLong(COLUMN_INDEX_ID),
                campaignId = campaignId,
                isClicked = cursor.getInt(COLUMN_INDEX_MSG_CLICKED),
                tag = cursor.getString(COLUMN_INDEX_MSG_TAG),
                receivedTime = milliSecondsFromIsoString(ISO8601Utils.format(Date(cursor.getLong(COLUMN_INDEX_GTIME)))),
                expiry = milliSecondsFromIsoString(ISO8601Utils.format(Date(cursor.getLong(COLUMN_INDEX_MSG_TTL)))),
                payload = messageJson.toString()
            )
        } catch (e: Exception) {
            Logger.print(LogLevel.ERROR, e) { "$tag cursorToInboxEntity() : " }
        }
        return null
    }
}