/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.richnotification.internal

import android.net.Uri
import com.moengage.core.PUSH_NOTIFICATION_COUPON_CODE
import com.moengage.core.PUSH_NOTIFICATION_NAVIGATION_ACTIVITY_NAME
import com.moengage.core.PUSH_NOTIFICATION_NAVIGATION_DEEPLINK
import com.moengage.core.PUSH_NOTIFICATION_NAVIGATION_DEEPLINK_LEGACY
import com.moengage.core.PUSH_NOTIFICATION_SHOW_DIALOG
import com.moengage.core.PUSH_NOTIFICATION_TYPE
import com.moengage.core.PUSH_NOTIFICATION_TYPE_DEEP_LINK_NOTIFICATION
import com.moengage.core.PUSH_NOTIFICATION_TYPE_NORMAL_NOTIFICATION
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.getEncodedDeepLinkString
import com.moengage.pushbase.ACTION_COUPON
import com.moengage.pushbase.ACTION_NAVIGATE
import com.moengage.pushbase.NAVIGATION_TYPE_DEEP_LINK
import com.moengage.pushbase.NAVIGATION_TYPE_RICH_LANDING
import com.moengage.pushbase.NAVIGATION_TYPE_SCREEN_NAME
import com.moengage.pushbase.internal.model.NotificationMetaData
import com.moengage.pushbase.model.action.Action
import com.moengage.pushbase.model.action.CouponAction
import com.moengage.pushbase.model.action.NavigateAction
import com.moengage.richnotification.internal.models.Template

/**
 * @author Umang Chamaria
 */

private const val tag = "${MODULE_TAG}DefaultActionUpdater"

@Throws(IllegalStateException::class)
internal fun updateDefaultAction(
    template: Template,
    metaData: NotificationMetaData,
    sdkInstance: SdkInstance
) {
    check(template.defaultAction.isNotEmpty()) { "Default action cannot be empty." }
    // remove action for default view.
    metaData.payload.payload.apply {
        remove(PUSH_NOTIFICATION_TYPE)
        remove(PUSH_NOTIFICATION_NAVIGATION_ACTIVITY_NAME)
        remove(PUSH_NOTIFICATION_NAVIGATION_DEEPLINK_LEGACY)
        remove(PUSH_NOTIFICATION_NAVIGATION_DEEPLINK)
        remove(PUSH_NOTIFICATION_SHOW_DIALOG)
        remove(PUSH_NOTIFICATION_COUPON_CODE)
    }
    for (action in template.defaultAction) {
        when (action.actionType) {
            ACTION_NAVIGATE -> updateNavigationAction(
                metaData,
                action,
                sdkInstance
            )
            ACTION_COUPON -> updateCouponActionInPayload(
                metaData,
                action,
                sdkInstance
            )
            else -> sdkInstance.logger.log { "$tag updateDefaultAction() : Not a valid default action." }
        }
    }
}

@Throws(IllegalStateException::class)
private fun updateNavigationAction(
    metaData: NotificationMetaData,
    action: Action,
    sdkInstance: SdkInstance
) {
    if (action !is NavigateAction) return
    sdkInstance.logger.log { "$tag updateNavigationAction() : Navigation Action: $action" }
    check(action.navigationUrl.isNotBlank()) { "Navigation url cannot be null" }
    when (action.navigationType) {
        NAVIGATION_TYPE_SCREEN_NAME -> {
            metaData.payload.payload.apply {
                putString(
                    PUSH_NOTIFICATION_TYPE,
                    PUSH_NOTIFICATION_TYPE_NORMAL_NOTIFICATION
                )
                putString(
                    PUSH_NOTIFICATION_NAVIGATION_ACTIVITY_NAME,
                    action.navigationUrl
                )
            }
            if (action.keyValue != null) {
                metaData.payload.payload.putAll(action.keyValue)
            }
        }
        NAVIGATION_TYPE_DEEP_LINK -> {
            metaData.payload.payload.putString(
                PUSH_NOTIFICATION_TYPE,
                PUSH_NOTIFICATION_TYPE_DEEP_LINK_NOTIFICATION
            )
            val link = Uri.parse(getEncodedDeepLinkString(action.navigationUrl))
            val builder = link.buildUpon()
            val kvPair = action.keyValue
            if (kvPair != null) {
                for (key in kvPair.keySet()) {
                    val value = kvPair[key]
                    if (value != null) builder.appendQueryParameter(key, value.toString())
                }
            }
            metaData.payload.payload.putString(
                PUSH_NOTIFICATION_NAVIGATION_DEEPLINK,
                builder.build().toString()
            )
        }
        NAVIGATION_TYPE_RICH_LANDING -> {
            metaData.payload.payload.apply {
                putString(
                    PUSH_NOTIFICATION_TYPE,
                    PUSH_NOTIFICATION_TYPE_NORMAL_NOTIFICATION
                )
                putString(
                    PUSH_NOTIFICATION_NAVIGATION_DEEPLINK_LEGACY,
                    action.navigationUrl
                )
                putString(
                    PUSH_NOTIFICATION_NAVIGATION_ACTIVITY_NAME,
                    "com.moe.pushlibrary.activities.MoEActivity"
                )
            }
        }
    }
}

private fun updateCouponActionInPayload(
    metaData: NotificationMetaData,
    action: Action,
    sdkInstance: SdkInstance
) {
    if (action !is CouponAction) return
    sdkInstance.logger.log { "$tag updateCouponActionInPayload() : Coupon Action: $action" }
    metaData.payload.payload.apply {
        putString(PUSH_NOTIFICATION_SHOW_DIALOG, "true")
        putString(PUSH_NOTIFICATION_COUPON_CODE, action.couponCode)
    }
}