/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.richnotification.internal

import android.os.Build
import android.os.Bundle
import com.moengage.core.LogLevel
import com.moengage.core.internal.logger.Logger
import com.moengage.pushbase.internal.PAYLOAD_ATTRIBUTE_MOE_FEATURES
import com.moengage.pushbase.internal.PAYLOAD_ATTRIBUTE_RICH_PUSH
import com.moengage.pushbase.model.NotificationPayload
import com.moengage.richnotification.internal.models.Card
import com.moengage.richnotification.internal.models.DefaultText
import org.json.JSONObject

/**
 * @author Umang Chamaria
 * Date: 03/03/20
 */
internal class Evaluator constructor(private val logger: Logger) {

    private val tag = "${MODULE_TAG}Evaluator"

    fun isTemplateSupported(payload: NotificationPayload): Boolean {
        val collapsedType = getCollapsedType(payload.payload)
        val expandedType = getExpandedType(payload.payload)
        return isTemplateSupported(collapsedType, expandedType, Build.VERSION.SDK_INT)
    }

    private fun getCollapsedType(payload: Bundle): String? {
        try {
            val payloadString =
                payload.getString(PAYLOAD_ATTRIBUTE_MOE_FEATURES)
                    ?: return null
            val payloadJson = JSONObject(payloadString)
            val templateJson =
                payloadJson.getJSONObject(PAYLOAD_ATTRIBUTE_RICH_PUSH)
            return if (!templateJson.has(COLLAPSED_CUSTOMISATION)) {
                null
            } else {
                templateJson.getJSONObject(COLLAPSED_CUSTOMISATION)
                    .getString(TEMPLATE_TYPE)
            }
        } catch (t: Throwable) {
            logger.log(LogLevel.ERROR, t) { "$tag getCollapsedType() : " }
        }
        return null
    }

    private fun getExpandedType(payload: Bundle): String? {
        try {
            val payloadString =
                payload.getString(PAYLOAD_ATTRIBUTE_MOE_FEATURES) ?: return null
            val payloadJson = JSONObject(payloadString)
            val templateJson = payloadJson.getJSONObject(PAYLOAD_ATTRIBUTE_RICH_PUSH)
            return if (!templateJson.has(EXPANDED_CUSTOMISATION)) {
                null
            } else {
                templateJson.getJSONObject(EXPANDED_CUSTOMISATION)
                    .getString(TEMPLATE_TYPE)
            }
        } catch (t: Throwable) {
            logger.log(LogLevel.ERROR, t) { "$tag getExpandedType() : " }
        }
        return null
    }

    fun hasMinimumText(defaultText: DefaultText): Boolean {
        return defaultText.title.isNotBlank() && defaultText.message.isNotBlank()
    }

    fun isValidBannerTextTemplate(card: Card): Boolean {
        for (widget in card.widgets) {
            if (widget.id == 0 && WIDGET_TYPE_IMAGE == widget.type) return true
        }
        return false
    }

    internal fun isTimerTemplate(
        collapsedTemplateType: String?,
        expandedTemplateType: String?
    ): Boolean {
        if (collapsedTemplateType == null || expandedTemplateType == null) return false
        return collapsedTemplateType == TEMPLATE_NAME_TIMER ||
            expandedTemplateType == TEMPLATE_NAME_TIMER ||
            collapsedTemplateType == TEMPLATE_NAME_TIMER_WITH_PROGRESS ||
            expandedTemplateType == TEMPLATE_NAME_TIMER_WITH_PROGRESS
    }

    internal fun isTimerWithProgressbarTemplate(
        collapsedTemplateType: String?,
        expandedTemplateType: String?
    ): Boolean {
        if (collapsedTemplateType == null || expandedTemplateType == null) return false
        return collapsedTemplateType == TEMPLATE_NAME_TIMER_WITH_PROGRESS ||
            expandedTemplateType == TEMPLATE_NAME_TIMER_WITH_PROGRESS
    }

    @Suppress("SameParameterValue")
    internal fun isTemplateSupported(
        collapsedTemplateType: String?,
        expandedTemplateType: String?,
        buildSdkVersion: Int
    ): Boolean {
        if (collapsedTemplateType.isNullOrBlank() && expandedTemplateType.isNullOrBlank()) return false

        return (
            (
                expandedTemplateType.isNullOrBlank() &&
                    SUPPORTED_COLLAPSED_STATES.contains(collapsedTemplateType)
                ) ||
                (
                    collapsedTemplateType.isNullOrBlank() &&
                        SUPPORTED_EXPANDED_STATES.contains(expandedTemplateType)
                    ) ||
                (
                    SUPPORTED_COLLAPSED_STATES.contains(collapsedTemplateType) ||
                        SUPPORTED_EXPANDED_STATES.contains(expandedTemplateType)
                    )
            ) &&
            isTemplateSupportedOnDevice(collapsedTemplateType, expandedTemplateType, buildSdkVersion)
    }

    private fun isTemplateSupportedOnDevice(
        collapsedTemplateType: String?,
        expandedTemplateType: String?,
        buildSdkVersion: Int
    ): Boolean {
        return if (collapsedTemplateType == TEMPLATE_NAME_TIMER ||
            expandedTemplateType == TEMPLATE_NAME_TIMER ||
            collapsedTemplateType == TEMPLATE_NAME_TIMER_WITH_PROGRESS ||
            expandedTemplateType == TEMPLATE_NAME_TIMER_WITH_PROGRESS
        ) {
            buildSdkVersion >= Build.VERSION_CODES.N
        } else {
            true
        }
    }
}