package com.moengage.richnotification.internal

import android.app.NotificationManager
import android.content.Context
import android.os.Build
import android.os.Bundle
import androidx.annotation.RequiresApi
import com.moengage.core.LogLevel
import com.moengage.core.PUSH_NOTIFICATION_CAMPAIGN_ID
import com.moengage.core.internal.executor.Job
import com.moengage.core.internal.global.GlobalResources
import com.moengage.core.internal.global.notifyPreProcessListenerIfRequired
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.SdkInstance
import com.moengage.pushbase.MOE_NOTIFICATION_ID
import com.moengage.pushbase.internal.KEY_RE_NOTIFY
import com.moengage.pushbase.internal.PushHelper

/**
 * @author Arshiya Khanum
 */
private const val TAG_MOE_RICH_PUSH_INTENT_ACTION_TASK = "MOE_RICH_PUSH_INTENT_ACTION_TASK"

@RequiresApi(Build.VERSION_CODES.M)
internal class IntentActionHandler(
    private val context: Context,
    private val intentAction: String,
    private val payload: Bundle
) {
    private val tag = "${MODULE_TAG}IntentActionHandler"

    fun handleAction() {
        try {
            GlobalResources.executor.submit {
                notifyPreProcessListenerIfRequired(payload)
                val instance =
                    PushHelper.getInstance().getSdkInstanceForPayload(payload) ?: return@submit
                instance.taskHandler.submit(
                    Job(TAG_MOE_RICH_PUSH_INTENT_ACTION_TASK, false) {
                        instance.logger.log { "$tag handleAction(): will process $intentAction" }
                        when (intentAction) {
                            INTENT_ACTION_PROGRESS_UPDATE -> handleProgressUpdateAction(
                                context,
                                instance,
                                payload
                            )
                            INTENT_ACTION_TIMER_ON_EXPIRY -> handleTimerExpiryAction(
                                context,
                                instance,
                                payload
                            )
                        }
                    }
                )
            }
        } catch (t: Throwable) {
            Logger.print(LogLevel.ERROR, t) { "$tag handleAction() : " }
        }
    }

    private fun handleProgressUpdateAction(
        context: Context,
        sdkInstance: SdkInstance,
        payload: Bundle
    ) {
        sdkInstance.logger.log {
            "$tag handleProgressUpdateAction() : will update progress value " +
                "in the notification"
        }
        val notificationId = payload.getInt(MOE_NOTIFICATION_ID)
        val notificationPayload = getCampaignPayload(
            payload.getString(PUSH_NOTIFICATION_CAMPAIGN_ID),
            context,
            sdkInstance
        ) ?: return

        val notificationManager =
            context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        val isNotificationActive =
            notificationManager.activeNotifications.find { notification -> notification.id == notificationId } != null
        if (!isNotificationActive) {
            sdkInstance.logger.log {
                "$tag handleProgressUpdateAction() : " +
                    "Notification: $notificationId is in dismissed state, cancelling the progress update."
            }
            cancelAlarmIfAny(context, payload, sdkInstance)
            return
        }
        notificationPayload.putBoolean(KEY_RE_NOTIFY, true)
        PushHelper.getInstance().handlePushPayload(context, notificationPayload)
    }

    private fun handleTimerExpiryAction(
        context: Context,
        sdkInstance: SdkInstance,
        payload: Bundle
    ) {
        sdkInstance.logger.log { "$tag handleTimerExpiryAction() : " }

        val templateName = payload.getString(TEMPLATE_NAME) ?: return
        val notificationId = payload.getInt(MOE_NOTIFICATION_ID)
        val notificationPayload = getCampaignPayload(
            payload.getString(PUSH_NOTIFICATION_CAMPAIGN_ID),
            context,
            sdkInstance
        ) ?: return

        cancelAlarmIfAny(context, payload, sdkInstance)

        dismissNotification(
            context,
            notificationPayload,
            templateName,
            notificationId,
            sdkInstance
        )
    }

    private fun dismissNotification(
        context: Context,
        payload: Bundle,
        templateName: String,
        notificationId: Int,
        sdkInstance: SdkInstance
    ) {
        sdkInstance.logger.log {
            "$tag dismissNotification() : notificationId: $notificationId, " +
                "templateName: $templateName"
        }
        dismissNotificationOnTimerExpiry(
            context,
            payload,
            templateName,
            notificationId,
            sdkInstance
        )
    }

    private fun getCampaignPayload(
        campaignId: String?,
        context: Context,
        sdkInstance: SdkInstance
    ): Bundle? {
        if (campaignId == null) return null
        return PushHelper.getInstance()
            .getCampaignPayloadForCampaignId(context, sdkInstance, campaignId)
    }
}