package com.moengage.richnotification.internal

import android.annotation.SuppressLint
import android.app.AlarmManager
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import android.os.Bundle
import androidx.annotation.RequiresApi
import androidx.core.app.NotificationCompat
import androidx.core.content.ContextCompat
import com.moengage.core.MOENGAGE_ACCOUNT_IDENTIFIER
import com.moengage.core.PUSH_NOTIFICATION_CAMPAIGN_ID
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.currentMillis
import com.moengage.core.internal.utils.getPendingIntentBroadcast
import com.moengage.core.internal.utils.getUniqueNumber
import com.moengage.pushbase.MOE_NOTIFICATION_ID
import com.moengage.pushbase.internal.KEY_RE_NOTIFY
import com.moengage.pushbase.internal.NOTIFICATION_REPOSTING_SOURCE
import com.moengage.pushbase.internal.NOTIFICATION_REPOSTING_SOURCE_REMIND_LATER_OR_SNOOZE
import com.moengage.pushbase.internal.PushHelper
import com.moengage.pushbase.internal.TEMPLATE_META
import com.moengage.pushbase.internal.model.NotificationMetaData
import com.moengage.pushbase.internal.model.TemplateTrackingMeta
import com.moengage.pushbase.internal.templateTrackingMetaToJsonString
import com.moengage.richnotification.internal.models.ProgressProperties
import com.moengage.richnotification.internal.models.Template
import com.moengage.richnotification.internal.models.TimerProperties
import com.moengage.richnotification.internal.models.TimerTemplate

/**
 * @author Arshiya Khanum
 */

private const val tag = "${MODULE_TAG}RichPushTimerUtils"

/**
 * Returns the timer end time
 *
 * @param template [Template]
 * @return if template has timer return end time milliseconds, else -1
 */
internal fun getTimerEndTime(template: Template): ProgressProperties {
    return if (template is TimerTemplate) {
        ProgressProperties(
            getTimerEndTime(
                template.timerProperties.duration,
                template.timerProperties.expiry
            ),
            template.timerProperties
        )
    } else {
        ProgressProperties(-1L, TimerProperties(-1, -1))
    }
}

internal fun getTimerEndTime(duration: Long, endTime: Long): Long {
    if (duration < TIMER_MIN_DURATION || duration > TIMER_MAX_DURATION) {
        return -1
    }
    val durationInMillis = duration * 1000
    val endTimeInMillis = endTime * 1000
    val availableDuration = endTimeInMillis - currentMillis()
    return if (availableDuration <= TIMER_MIN_DURATION_LEFT) {
        -1
    } else if (availableDuration < durationInMillis) {
        availableDuration
    } else {
        durationInMillis
    }
}

@RequiresApi(Build.VERSION_CODES.N)
internal fun getTimerExpiryIntent(
    context: Context,
    metaData: NotificationMetaData,
    template: TimerTemplate,
    progressProperties: ProgressProperties
): PendingIntent {
    val expiryIntent = Intent(context, MoERichPushReceiver::class.java)
    metaData.payload.payload.apply {
        putInt(MOE_NOTIFICATION_ID, metaData.notificationId)
        putString(TEMPLATE_NAME, template.templateName)
    }
    expiryIntent.apply {
        flags = Intent.FLAG_ACTIVITY_NEW_TASK
        putExtra(MOE_NOTIFICATION_ID, metaData.notificationId)
        putExtra(TIMER_ALARM_ID, progressProperties.timerAlarmId)
        putExtra(TEMPLATE_NAME, template.templateName)
        putExtra(PUSH_NOTIFICATION_CAMPAIGN_ID, metaData.payload.campaignId)
        putExtra(
            MOENGAGE_ACCOUNT_IDENTIFIER,
            metaData.payload.payload.getString(MOENGAGE_ACCOUNT_IDENTIFIER)
        )
        action = INTENT_ACTION_TIMER_ON_EXPIRY
    }

    return getPendingIntentBroadcast(context, progressProperties.timerAlarmId, expiryIntent)
}

@SuppressLint("MissingPermission")
@RequiresApi(Build.VERSION_CODES.N)
internal fun setTimerExpiryAlarm(
    context: Context,
    template: Template,
    metaData: NotificationMetaData,
    progressProperties: ProgressProperties,
    expiryTriggerInMillis: Long
) {
    template as TimerTemplate

    if (!hasScheduleExactPermission(context)) return
    val intent = getTimerExpiryIntent(context, metaData, template, progressProperties)
    val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager
    alarmManager.setExactAndAllowWhileIdle(
        AlarmManager.RTC_WAKEUP,
        expiryTriggerInMillis,
        intent
    )
    Logger.print { "setTimerExpiryAlarm() : progressProperties: $progressProperties" }
}

internal fun updateNotificationBuilderForTimerTemplate(
    notificationBuilder: NotificationCompat.Builder,
    progressProperties: ProgressProperties
) {
    notificationBuilder.setStyle(NotificationCompat.DecoratedCustomViewStyle())
        // Remove the large icon if any was added in the basic notification.
        // DecoratedCustomViewStyle accepts the large icon and displays it. Setting it explicitly
        // to `null` to avoid the broken layout design.
        .setLargeIcon(null)
        // Summary is not supported in Timer and Timer with Progress bar templates
        .setSubText(null)
        // Auto dismisses the notification at the timer end time
        .setTimeoutAfter(progressProperties.timerEndTime)
}

internal fun cancelAlarmIfAny(context: Context, bundle: Bundle, sdkInstance: SdkInstance) {
    sdkInstance.logger.log { "$tag cancelAlarmIfAny(): " }
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
        cancelTimerAlarmIfAny(context, bundle, sdkInstance)
        cancelProgressAlarmIfAny(context, bundle, sdkInstance)
    }
}

@RequiresApi(Build.VERSION_CODES.N)
internal fun cancelTimerAlarmIfAny(context: Context, bundle: Bundle, sdkInstance: SdkInstance) {
    val notificationId = bundle.get(MOE_NOTIFICATION_ID)
    val timerAlarmId = bundle.getInt(TIMER_ALARM_ID)

    sdkInstance.logger.log { "$tag cancelTimerAlarmIfAny(): notificationId:$notificationId, timerAlarmId: $timerAlarmId" }
    val expiryIntent = Intent(context, MoERichPushReceiver::class.java)
    expiryIntent.apply {
        flags = Intent.FLAG_ACTIVITY_NEW_TASK
        putExtra(MOE_NOTIFICATION_ID, bundle.getInt(MOE_NOTIFICATION_ID))
        putExtra(TIMER_ALARM_ID, bundle.getInt(TIMER_ALARM_ID))
        putExtra(TEMPLATE_NAME, bundle.getString(TEMPLATE_NAME))
        putExtra(PUSH_NOTIFICATION_CAMPAIGN_ID, bundle.getString(PUSH_NOTIFICATION_CAMPAIGN_ID))
        putExtra(MOENGAGE_ACCOUNT_IDENTIFIER, bundle.getString(MOENGAGE_ACCOUNT_IDENTIFIER))
        action = INTENT_ACTION_TIMER_ON_EXPIRY
    }

    val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager
    alarmManager.cancel(getPendingIntentBroadcast(context, timerAlarmId, expiryIntent))
}

@RequiresApi(Build.VERSION_CODES.N)
internal fun cancelProgressAlarmIfAny(context: Context, bundle: Bundle, sdkInstance: SdkInstance) {
    val notificationId = bundle.getInt(MOE_NOTIFICATION_ID)
    val progressAlarmId = bundle.getInt(PROGRESS_ALARM_ID)

    sdkInstance.logger.log { "$tag cancelProgressAlarmIfAny(): notificationId:$notificationId, progressAlarmId: $progressAlarmId" }
    val expiryIntent = Intent(context, MoERichPushReceiver::class.java)
    expiryIntent.apply {
        flags = Intent.FLAG_ACTIVITY_NEW_TASK
        putExtra(MOE_NOTIFICATION_ID, bundle.getInt(MOE_NOTIFICATION_ID))
        putExtra(PUSH_NOTIFICATION_CAMPAIGN_ID, bundle.getString(PUSH_NOTIFICATION_CAMPAIGN_ID))
        putExtra(TEMPLATE_NAME, bundle.getString(TEMPLATE_NAME))
        putExtra(PROGRESS_ALARM_ID, progressAlarmId)
        putExtra(MOENGAGE_ACCOUNT_IDENTIFIER, bundle.getString(MOENGAGE_ACCOUNT_IDENTIFIER))
        action = INTENT_ACTION_PROGRESS_UPDATE
    }

    val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager
    alarmManager.cancel(getPendingIntentBroadcast(context, progressAlarmId, expiryIntent))
}

/**
 * Cancel the notification on timer duration expiry
 * NOTE: This api call should not track dismiss event
 */
internal fun dismissNotificationOnTimerExpiry(
    context: Context,
    payload: Bundle,
    templateName: String,
    notificationId: Int,
    sdkInstance: SdkInstance
) {
    if (Build.VERSION.SDK_INT == Build.VERSION_CODES.N) {
        val manager =
            context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        manager.cancel(notificationId)
    }
    payload.apply {
        putString(
            TEMPLATE_META,
            templateTrackingMetaToJsonString(
                TemplateTrackingMeta(
                    templateName,
                    -1,
                    -1
                )
            )
        )
        putInt(MOE_NOTIFICATION_ID, notificationId)
    }

    PushHelper.getInstance().handleNotificationCancelled(context, payload, sdkInstance)
}

@Suppress("ConvertTwoComparisonsToRangeCheck")
internal fun setProgressUpdateProperties(
    progressProperties: ProgressProperties,
    sdkInstance: SdkInstance
): ProgressProperties {
    val actualDuration = progressProperties.timerProperties.duration
    val elapsedDuration = actualDuration - progressProperties.timerEndTime / 1000
    val progressUpdateCount: Int
    val progressUpdateValue: Int

    if (actualDuration >= TIMER_MIN_DURATION && actualDuration <=
        PROGRESS_BAR_TEMPLATE_MAX_TOTAL_DURATION
    ) {
        progressUpdateCount = 10
        progressUpdateValue = 10
    } else if (actualDuration > PROGRESS_BAR_TEMPLATE_MAX_TOTAL_DURATION && actualDuration <=
        TIMER_MAX_DURATION
    ) {
        progressUpdateCount = 25
        progressUpdateValue = 4
    } else {
        sdkInstance.logger.log {
            "setProgressUpdateProperties() : total Duration  left is in not" +
                "in the range of 15 minutes to 12 hours."
        }
        progressUpdateCount = -1
        progressUpdateValue = -1
    }
    if (progressUpdateCount != -1 && progressUpdateValue != -1) {
        val progressUpdateInterval = actualDuration / progressUpdateCount
        val currentProgress =
            ((elapsedDuration / progressUpdateInterval) * progressUpdateValue).toInt()

        progressProperties.setProgressUpdateParameters(
            progressUpdateInterval * 1000,
            progressUpdateValue,
            currentProgress,
            progressUpdateCount,
            currentProgress / progressUpdateCount
        )
    }
    sdkInstance.logger.log { "setProgressUpdateProperties() : $progressProperties" }
    return progressProperties
}

@SuppressLint("MissingPermission")
@RequiresApi(Build.VERSION_CODES.N)
internal fun scheduleProgressTemplateUpdateAlarm(
    context: Context,
    template: Template,
    metaData: NotificationMetaData,
    progressProperties: ProgressProperties
) {
    template as TimerTemplate

    if (!hasScheduleExactPermission(context)) return
    val intent = getProgressUpdateIntent(context, metaData, template, progressProperties)
    val alarmManager = context.getSystemService(Context.ALARM_SERVICE) as AlarmManager

    alarmManager.setExactAndAllowWhileIdle(
        AlarmManager.RTC_WAKEUP,
        currentMillis() + progressProperties.updateInterval,
        intent
    )
}

@RequiresApi(Build.VERSION_CODES.N)
internal fun getProgressUpdateIntent(
    context: Context,
    metaData: NotificationMetaData,
    template: TimerTemplate,
    progressProperties: ProgressProperties
): PendingIntent {
    val intent = Intent(context, MoERichPushReceiver::class.java)
    metaData.payload.payload.apply {
        putInt(MOE_NOTIFICATION_ID, metaData.notificationId)
        putString(TEMPLATE_NAME, template.templateName)
        putInt(
            PROGRESS_BAR_TEMPLATE_CURRENT_PROGRESS_VALUE,
            progressProperties.currentProgress +
                progressProperties.progressIncrementPercent
        )
        putInt(
            PROGRESS_BAR_TEMPLATE_PROGRESS_INCREMENT_VALUE,
            progressProperties.progressIncrementPercent
        )
        putLong(PROGRESS_BAR_TEMPLATE_PROGRESS_UPDATE_INTERVAL, progressProperties.updateInterval)
        putInt(
            PROGRESS_BAR_TEMPLATE_MAX_PROGRESS_UPDATES_COUNT,
            progressProperties.maxUpdatesCount
        )
        putInt(
            PROGRESS_BAR_TEMPLATE_CURRENT_PROGRESS_UPDATES_COUNT,
            progressProperties.currentUpdatesCount + 1
        )
    }
    intent.apply {
        flags = Intent.FLAG_ACTIVITY_NEW_TASK
        putExtra(MOE_NOTIFICATION_ID, metaData.notificationId)
        putExtra(PUSH_NOTIFICATION_CAMPAIGN_ID, metaData.payload.campaignId)
        putExtra(TEMPLATE_NAME, template.templateName)
        putExtra(PROGRESS_ALARM_ID, progressProperties.progressAlarmId)
        putExtra(
            MOENGAGE_ACCOUNT_IDENTIFIER,
            metaData.payload.payload.getString(MOENGAGE_ACCOUNT_IDENTIFIER)
        )
        action = INTENT_ACTION_PROGRESS_UPDATE
    }

    return getPendingIntentBroadcast(context, progressProperties.progressAlarmId, intent)
}

internal fun setUpTimerAndProgressComponents(
    context: Context,
    template: Template,
    metaData: NotificationMetaData,
    sdkInstance: SdkInstance,
    progressProperties: ProgressProperties
) {
    if (Build.VERSION.SDK_INT < Build.VERSION_CODES.N) return
    // This is required to make sure the header is displayed
    updateNotificationBuilderForTimerTemplate(metaData.notificationBuilder, progressProperties)
    if (progressProperties.timerEndTime == -1L) {
        sdkInstance.logger.log { "$tag setUpTimerAndProgressComponents(): endTime is -1" }
        return
    }

    val expiryTriggerInMillis = currentMillis() + progressProperties.timerEndTime
    setUpTimerComponentsIfRequired(
        context,
        template,
        metaData,
        progressProperties,
        expiryTriggerInMillis
    )

    setupProgressbarComponentsIfRequired(
        context,
        template,
        metaData,
        progressProperties,
        sdkInstance
    )

    PushHelper.getInstance()
        .storeRepostCampaignPayload(context, sdkInstance, metaData.payload, expiryTriggerInMillis)
}

@RequiresApi(Build.VERSION_CODES.N)
private fun setUpTimerComponentsIfRequired(
    context: Context,
    template: Template,
    metaData: NotificationMetaData,
    progressProperties: ProgressProperties,
    expiryTriggerInMillis: Long
) {
    Logger.print {
        "$tag: setUpTimerComponentsIfRequired(): notificationId: ${metaData.notificationId}, " +
            "alarmId: ${progressProperties.timerAlarmId}, triggerInMillis: " +
            "${progressProperties.timerEndTime}"
    }
    if (metaData.payload.payload.getBoolean(KEY_RE_NOTIFY)) return
    setTimerExpiryAlarm(
        context,
        template,
        metaData,
        progressProperties,
        expiryTriggerInMillis
    )
}

private fun setupProgressbarComponentsIfRequired(
    context: Context,
    template: Template,
    metaData: NotificationMetaData,
    progressProperties: ProgressProperties,
    sdkInstance: SdkInstance
) {
    if (Build.VERSION.SDK_INT < Build.VERSION_CODES.N || !Evaluator(sdkInstance.logger)
        .isTimerWithProgressbarTemplate(
                template.collapsedTemplate?.type,
                template.expandedTemplate?.type
            )
    ) {
        return
    }

    if (progressProperties.currentUpdatesCount == progressProperties.maxUpdatesCount - 1) {
        progressProperties.updateInterval = progressProperties.timerEndTime
    }
    Logger.print {
        "$tag setupProgressbarComponentsIfRequired(): notificationId: ${metaData.notificationId}, " +
            "progressProperties: $progressProperties"
    }
    scheduleProgressTemplateUpdateAlarm(context, template, metaData, progressProperties)
}

internal fun addProgressPropertiesIfRequired(
    progressProperties: ProgressProperties,
    template: Template,
    metaData: NotificationMetaData,
    sdkInstance: SdkInstance
): ProgressProperties {
    if (template is TimerTemplate && Evaluator(sdkInstance.logger)
        .isTimerWithProgressbarTemplate(
                template.collapsedTemplate?.type,
                template.expandedTemplate?.type
            ) && progressProperties.timerEndTime > -1L
    ) {
        if (metaData.payload.payload.getBoolean(KEY_RE_NOTIFY) &&
            !metaData.payload.payload.getString(NOTIFICATION_REPOSTING_SOURCE, "")
                .equals(NOTIFICATION_REPOSTING_SOURCE_REMIND_LATER_OR_SNOOZE)
        ) {
            progressProperties.setProgressUpdateParameters(
                metaData.payload.payload.getInt(PROGRESS_BAR_TEMPLATE_PROGRESS_UPDATE_INTERVAL)
                    .toLong(),
                metaData.payload.payload.getInt(PROGRESS_BAR_TEMPLATE_PROGRESS_INCREMENT_VALUE),
                metaData.payload.payload.getInt(PROGRESS_BAR_TEMPLATE_CURRENT_PROGRESS_VALUE),
                metaData.payload.payload.getInt(PROGRESS_BAR_TEMPLATE_MAX_PROGRESS_UPDATES_COUNT),
                metaData.payload.payload.getInt(
                    PROGRESS_BAR_TEMPLATE_CURRENT_PROGRESS_UPDATES_COUNT
                )
            )
        } else {
            setProgressUpdateProperties(progressProperties, sdkInstance)
            // Removing key from bundle so that next time calculated value can be reused
            metaData.payload.payload.remove(NOTIFICATION_REPOSTING_SOURCE)
        }
    }
    return progressProperties
}

/**
 * Use the same alarm id for progress update event.
 */
internal fun getProgressUpdateAlarmId(metaData: NotificationMetaData): Int {
    return if (metaData.payload.payload.getBoolean(KEY_RE_NOTIFY)) {
        metaData.payload.payload.getInt(PROGRESS_ALARM_ID)
    } else {
        getUniqueNumber()
    }
}

internal fun getTimerExpiryAlarmId(metaData: NotificationMetaData): Int {
    return if (metaData.payload.payload.getBoolean(KEY_RE_NOTIFY)) {
        metaData.payload.payload.getInt(TIMER_ALARM_ID)
    } else {
        getUniqueNumber()
    }
}

/**
 * Check if Application has the permission to Schedule exact alarm or not.
 * For Android Sdk Version < S, Application can always schedule the exact alarm,
 * else check if Application has the permission [android.Manifest.permission.SCHEDULE_EXACT_ALARM]
 *
 * @param context instance of [Context]
 * @return true if Application can schedule exact alarms, else false
 */
internal fun hasScheduleExactPermission(context: Context): Boolean {
    val state = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
        val alarmManager = ContextCompat.getSystemService(context, AlarmManager::class.java)
        alarmManager?.canScheduleExactAlarms() ?: false
    } else {
        true
    }
    Logger.print { "$tag hasScheduleExactPermission() : $state" }
    return state
}