/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.richnotification.internal

import android.app.NotificationManager
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.os.Build
import android.text.Spanned
import android.text.format.DateFormat
import androidx.annotation.ChecksSdkIntAtLeast
import androidx.core.text.HtmlCompat
import com.moengage.core.LogLevel
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.remoteconfig.RemoteConfig
import com.moengage.core.internal.utils.MoEUtils
import com.moengage.core.internal.utils.getPendingIntentService
import com.moengage.pushbase.MOE_NOTIFICATION_ID
import com.moengage.pushbase.internal.INTENT_ACTION_NOTIFICATION_CLEARED
import com.moengage.pushbase.internal.MoEPushWorker
import com.moengage.pushbase.internal.NOTIFICATION_CLEARED_REQUEST_ID
import com.moengage.pushbase.internal.PushHelper
import com.moengage.pushbase.internal.TEMPLATE_META
import com.moengage.pushbase.internal.model.NotificationMetaData
import com.moengage.pushbase.internal.model.TemplateTrackingMeta
import com.moengage.pushbase.internal.templateTrackingMetaToJsonString
import com.moengage.pushbase.model.NotificationPayload
import com.moengage.richnotification.internal.models.Template
import java.util.*

/**
 * @author Umang Chamaria
 * Date: 03/03/20
 */
private const val tag = "${MODULE_TAG}RichPushUtils"

@Throws(PackageManager.NameNotFoundException::class)
internal fun getAppName(context: Context): String {
    val packageManager = context.packageManager
    val info = packageManager.getApplicationInfo(
        context.packageName,
        PackageManager.GET_META_DATA
    )
    return packageManager.getApplicationLabel(info) as String
}

internal fun getTime(): String {
    val dateFormat = "hh:mm aaa"
    return DateFormat.format(dateFormat, Calendar.getInstance().time) as String
}

internal fun scaleLandscapeBitmap(context: Context, imageBitmap: Bitmap): Bitmap {
    var bitmap: Bitmap = imageBitmap
    if (bitmap.width > bitmap.height) {
        val displayMetrics = context.resources.displayMetrics
        val height = bitmap.height * displayMetrics.widthPixels / bitmap.width
        try {
            bitmap =
                Bitmap.createScaledBitmap(bitmap, displayMetrics.widthPixels, height, true)
        } catch (t: Throwable) {
            Logger.print(LogLevel.ERROR, t) { "$tag scaleLandscapeBitmap() : " }
        }
    }
    return bitmap
}

internal fun getHtmlText(string: String): Spanned {
    return HtmlCompat.fromHtml(string, HtmlCompat.FROM_HTML_MODE_COMPACT)
}

// Big layout is 100/285dp
internal fun isBigLayoutSupported(remoteConfig: RemoteConfig): Boolean {
    return Build.VERSION.SDK_INT >= Build.VERSION_CODES.N &&
        remoteConfig.pushConfig.whiteListedOems.contains(
            MoEUtils.deviceManufacturer().uppercase()
        )
}

internal fun getTemplateLayout(layoutSmall: Int, layoutBig: Int, sdkInstance: SdkInstance): Int {
    return if (isBigLayoutSupported(sdkInstance.remoteConfig)) {
        sdkInstance.logger.log { "$tag getTemplateLayout() : Big layout selected" }
        layoutBig
    } else {
        sdkInstance.logger.log { "$tag getTemplateLayout() : Big small selected" }
        layoutSmall
    }
}

internal fun isPushTemplateSupported(
    payload: NotificationPayload,
    sdkInstance: SdkInstance
): Boolean {
    val evaluator = Evaluator(sdkInstance.logger)
    val isTemplateSupported = payload.addOnFeatures.isRichPush && evaluator.isTemplateSupported(
        payload
    )

    sdkInstance.logger.log { "$tag isTemplateSupported() : Template Supported? $isTemplateSupported" }
    return isTemplateSupported
}

internal fun setNotificationClearIntent(
    context: Context,
    metaData: NotificationMetaData,
    finalIntent: Intent
) {
    val intent = getPendingIntentService(
        context,
        metaData.notificationId or NOTIFICATION_CLEARED_REQUEST_ID,
        finalIntent
    )
    metaData.notificationBuilder.setDeleteIntent(intent)
}

internal fun getNotificationClearIntent(
    context: Context,
    metaData: NotificationMetaData,
    template: Template
): Intent {
    val finalIntent = Intent(context, MoEPushWorker::class.java)
    finalIntent.apply {
        putExtras(metaData.payload.payload)
        putExtra(
            TEMPLATE_META,
            templateTrackingMetaToJsonString(
                TemplateTrackingMeta(
                    template.templateName,
                    -1,
                    -1
                )
            )
        )
        putExtra(MOE_NOTIFICATION_ID, metaData.notificationId)
        action = INTENT_ACTION_NOTIFICATION_CLEARED
    }
    return finalIntent
}

internal fun handleLogout(
    context: Context,
    sdkInstance: SdkInstance
) {
    try {
        clearNotificationsAndCancelAlarms(context, sdkInstance)
    } catch (t: Throwable) {
        sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag onLogout() : " }
    }
}

@ChecksSdkIntAtLeast(api = Build.VERSION_CODES.S)
internal fun doesSdkSupportDecoratedStyleOnDevice(): Boolean {
    return Build.VERSION.SDK_INT >= Build.VERSION_CODES.S
}

/**
 * Clears notifications and cancels all the associated alarms
 * @since 4.4.0
 */
internal fun clearNotificationsAndCancelAlarms(context: Context, sdkInstance: SdkInstance) {
    val activeCampaigns =
        PushHelper.getInstance().getCampaignPayloadsForActiveCampaigns(context, sdkInstance)

    sdkInstance.logger.log { "$tag clearNotificationsAndCancelAlarms() : active template campaigns: ${activeCampaigns.size}" }
    val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as
        NotificationManager

    for (payload in activeCampaigns) {
        notificationManager.cancel(payload.getInt(MOE_NOTIFICATION_ID))
        cancelAlarmIfAny(context, payload, sdkInstance)
    }
}

/**
 * Get the notification layout based on the available height in notification for decorated style.
 * This has the check based on the [DECORATED_STYLE_SMALL_LAYOUT_MANUFACTURER]
 *
 * @author Abhishek Kumar
 * @since 4.7.0
 */
internal fun getDecoratedStyleTemplateLayout(
    smallLayout: Int,
    defaultLayout: Int,
    sdkInstance: SdkInstance
): Int {
    return if (DECORATED_STYLE_SMALL_LAYOUT_MANUFACTURER.contains(
            MoEUtils.deviceManufacturer().uppercase()
        )
    ) {
        sdkInstance.logger.log { "$tag getDecoratedStyleTemplateLayout(): Small layout selected" }
        smallLayout
    } else {
        sdkInstance.logger.log { "$tag getDecoratedStyleTemplateLayout(): Default layout selected" }
        defaultLayout
    }
}