/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.richnotification.internal.builder

import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.view.View
import android.widget.RemoteViews
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.downloadImageBitmap
import com.moengage.core.internal.utils.getPendingIntentService
import com.moengage.core.internal.utils.getUniqueNumber
import com.moengage.core.internal.utils.isTablet
import com.moengage.pushbase.MOE_NOTIFICATION_ID
import com.moengage.pushbase.internal.PAYLOAD_ATTRIBUTE_MOE_FEATURES
import com.moengage.pushbase.internal.PAYLOAD_ATTRIBUTE_RICH_PUSH
import com.moengage.pushbase.internal.isReNotification
import com.moengage.pushbase.internal.model.NotificationMetaData
import com.moengage.pushbase.internal.transformToPx
import com.moengage.richnotification.R
import com.moengage.richnotification.internal.EXPANDED_CUSTOMISATION
import com.moengage.richnotification.internal.Evaluator
import com.moengage.richnotification.internal.IMAGE_INDEX
import com.moengage.richnotification.internal.MAX_IMAGE_HEIGHT
import com.moengage.richnotification.internal.MODULE_TAG
import com.moengage.richnotification.internal.MoERichPushIntentService
import com.moengage.richnotification.internal.NAVIGATION_DIRECTION
import com.moengage.richnotification.internal.NAVIGATION_DIRECTION_NEXT
import com.moengage.richnotification.internal.NAVIGATION_DIRECTION_PREVIOUS
import com.moengage.richnotification.internal.TOTAL_IMAGE_COUNT
import com.moengage.richnotification.internal.WIDGET_TYPE_IMAGE
import com.moengage.richnotification.internal.doesSdkSupportDecoratedStyleOnDevice
import com.moengage.richnotification.internal.getAppName
import com.moengage.richnotification.internal.getTemplateLayout
import com.moengage.richnotification.internal.models.Card
import com.moengage.richnotification.internal.models.CardWidget
import com.moengage.richnotification.internal.models.ImageWidget
import com.moengage.richnotification.internal.models.Template
import com.moengage.richnotification.internal.repository.CARDS
import com.moengage.richnotification.internal.repository.ImageManager
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

/**
 * @author Umang Chamaria
 * Date: 2020/07/16
 */
internal class CarouselBuilder(
    private val context: Context,
    private val template: Template,
    private val metaData: NotificationMetaData,
    private val sdkInstance: SdkInstance
) {
    private val tag = "${MODULE_TAG}CarouselBuilder"

    private val templateHelper = TemplateHelper(sdkInstance)

    fun buildSimpleCarousel(): Boolean {
        try {
            if (template.expandedTemplate == null) return false
            if (!Evaluator(sdkInstance.logger).hasMinimumText(template.defaultText)) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag buildSimpleCarousel() : Does not have minimum text." }
                return false
            }
            sdkInstance.logger.log { "$tag buildSimpleCarousel() : Will attempt to build carousal notification." }
            sdkInstance.logger.log { "$tag buildSimpleCarousel() : Template: ${template.expandedTemplate}" }
            val remoteViews = getCarouselRemoteView(
                template.expandedTemplate.autoStart,
                metaData.payload.addOnFeatures.isPersistent
            )
            if (template.expandedTemplate.cards.isEmpty()) return false
            templateHelper.addLayoutStyle(
                template.expandedTemplate.layoutStyle,
                remoteViews,
                R.id.expandedRootView
            )
            // default text
            templateHelper.setDefaultTextAndStyle(
                remoteViews,
                template.defaultText,
                getAppName(context),
                template.headerStyle
            )
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                templateHelper.addDecoratedStyleBaseProperties(
                    remoteViews,
                    R.id.expandedRootView,
                    template,
                    metaData
                )
                if (metaData.payload.addOnFeatures.isPersistent) {
                    templateHelper.setDismissCtaCustomization(
                        remoteViews,
                        template.dismissCta
                    )
                }
            } else {
                templateHelper.setHeaderAssetsAndIcon(
                    context,
                    remoteViews,
                    template,
                    metaData
                )
            }
            templateHelper.addLargeIcon(remoteViews, template, metaData.payload)
            // persistent notification
            if (metaData.payload.addOnFeatures.isPersistent) {
                templateHelper.addActionToDismissCTA(remoteViews, context, metaData)
            }
            val imageUrls = carouselImageUrls
            if (imageUrls.isEmpty()) return false
            var successCount = 0
            if (!isReNotification(metaData.payload.payload)) {
                successCount = downloadAndSaveImages(imageUrls)
                if (successCount == 0) return false
                if (successCount != imageUrls.size) {
                    removeFailedImagesFromPayload()
                }
                metaData.payload.payload.putInt(
                    TOTAL_IMAGE_COUNT,
                    successCount
                )
            }
            if (template.expandedTemplate.autoStart) {
                buildAutoStartCarousel(
                    remoteViews,
                    successCount,
                    template.expandedTemplate.cards
                )
            } else {
                buildManualSimpleCarousel(remoteViews, template.expandedTemplate.cards)
            }
            templateHelper.addDefaultActionToNotificationClick(
                context,
                remoteViews,
                R.id.collapsedRootView,
                template,
                metaData
            )
            metaData.notificationBuilder.setCustomBigContentView(remoteViews)
            return true
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag buildSimpleCarousel() : " }
        }
        return false
    }

    @Throws(IllegalStateException::class)
    private fun buildAutoStartCarousel(
        remoteViews: RemoteViews,
        imageDownloadCount: Int,
        cards: List<Card>
    ) {
        sdkInstance.logger.log { "$tag buildAutoStartCarousel() : Building auto start carousel." }
        val viewId: Int
        val imageArray: Array<CardWidget>
        when (imageDownloadCount) {
            1 -> {
                viewId = R.id.card11
                imageArray = singleImage
            }
            2 -> {
                viewId = R.id.viewFlipperTwo
                imageArray = viewFlipperTwoImageIdArray
            }
            3 -> {
                viewId = R.id.viewFlipperThree
                imageArray = viewFlipperThreeImageIdArray
            }
            4 -> {
                viewId = R.id.viewFlipperFour
                imageArray = viewFlipperFourImageIdArray
            }
            5 -> {
                viewId = R.id.viewFlipperFive
                imageArray = viewFlipperFiveImageIdArray
            }
            else -> throw IllegalStateException("Not a valid state")
        }
        remoteViews.setViewVisibility(viewId, View.VISIBLE)
        var index = 0
        val fileManager = ImageManager(context, sdkInstance)
        var cardIndex = 0
        while (index < imageArray.size && cardIndex < cards.size) {
            val card = cards[cardIndex]
            sdkInstance.logger.log { "$tag buildAutoStartCarousel() : Building Card: $card" }
            val widget = card.widgets[0]
            check(WIDGET_TYPE_IMAGE == widget.type) { "Only image widgets are supported in carousel." }
            val imageUrl = widget.content
            var bitmap = fileManager.getImageFromUrl(metaData.payload.campaignId, imageUrl)
            if (bitmap == null) {
                cardIndex++
                continue
            }
            bitmap = templateHelper.scaleBitmap(
                context,
                bitmap,
                transformToPx(context, MAX_IMAGE_HEIGHT)
            )
            val maxImageHeight = transformToPx(context, MAX_IMAGE_HEIGHT)
            val widgetId = if (isTablet(context)) {
                imageArray[index].horizontalCenterCropImageId
            } else if (bitmap.height >= bitmap.width) {
                imageArray[index].verticalImageId
            } else {
                if (bitmap.height >= maxImageHeight) {
                    imageArray[index].horizontalCenterCropImageId
                } else {
                    imageArray[index].horizontalFitCenterImageId
                }
            }
            sdkInstance.logger.log {
                "$tag buildAutoStartCarousel() : Image Dimensions: Height: ${bitmap.height} Width: ${bitmap.width}"
            }
            remoteViews.setViewVisibility(widgetId, View.VISIBLE)
            remoteViews.setImageViewBitmap(widgetId, bitmap)
            templateHelper.addActionToImageWidget(
                context,
                metaData,
                template,
                remoteViews,
                widget as ImageWidget,
                card,
                widgetId,
                imageArray[index].cardId
            )
            cardIndex++
            index++
        }
    }

    private fun buildManualSimpleCarousel(remoteViews: RemoteViews, cards: List<Card>) {
        val index = metaData.payload.payload.getInt(IMAGE_INDEX, 0)
        val totalImages =
            metaData.payload.payload.getInt(TOTAL_IMAGE_COUNT, -1)
        if (totalImages == -1 || index > totalImages) return

        // clean up previous data
        metaData.payload.payload.apply {
            remove(IMAGE_INDEX)
            remove(NAVIGATION_DIRECTION)
        }
        // set new content.
        val fileManager = ImageManager(context, sdkInstance)
        val card = cards[index]
        val widget = card.widgets[0]
        check(WIDGET_TYPE_IMAGE == widget.type) { "Only image widgets support in carousel." }
        val bitmap =
            fileManager.getImageFromUrl(metaData.payload.campaignId, widget.content) ?: return
        templateHelper.addImageWidgetToTemplate(
            context,
            metaData,
            template,
            remoteViews,
            widget as ImageWidget,
            card,
            bitmap
        )
        if (totalImages > 1) {
            remoteViews.setViewVisibility(R.id.arrowRight, View.VISIBLE)
            remoteViews.setViewVisibility(R.id.arrowLeft, View.VISIBLE)
            setUpIndicator(remoteViews, totalImages, index)
            val nextIntent = getDirectionIntent(
                context,
                metaData.payload.payload,
                metaData.notificationId,
                NAVIGATION_DIRECTION_NEXT,
                index,
                totalImages
            )
            val pendingIntentNext =
                getPendingIntentService(context, getUniqueNumber(), nextIntent)
            remoteViews.setOnClickPendingIntent(R.id.arrowRight, pendingIntentNext)
            val previousIntent = getDirectionIntent(
                context,
                metaData.payload.payload,
                metaData.notificationId,
                NAVIGATION_DIRECTION_PREVIOUS,
                index,
                totalImages
            )
            val pendingIntentPrevious =
                getPendingIntentService(context, getUniqueNumber(), previousIntent)
            remoteViews.setOnClickPendingIntent(R.id.arrowLeft, pendingIntentPrevious)
        }
    }

    private fun setUpIndicator(
        remoteViews: RemoteViews,
        totalImages: Int,
        currentPosition: Int
    ) {
        if (totalImages < 2) return
        remoteViews.setViewVisibility(
            R.id.markerLayout,
            View.VISIBLE
        )
        if (totalImages > markerImageIdArray.size) return
        for (i in 0 until totalImages) {
            remoteViews.setViewVisibility(markerImageIdArray[i], View.VISIBLE)
            remoteViews.setImageViewResource(
                markerImageIdArray[i],
                R.drawable.moe_rich_push_other_items
            )
        }
        remoteViews.setImageViewResource(
            markerImageIdArray[currentPosition],
            R.drawable.moe_rich_push_current_position
        )
    }

    private val carouselImageUrls: List<String>
        get() {
            template.expandedTemplate?.cards ?: return emptyList()
            val imageUrls: MutableList<String> = ArrayList(template.expandedTemplate.cards.size)
            for (card in template.expandedTemplate.cards) {
                check(card.widgets.isNotEmpty()) { "Widget list should not be empty" }
                check(card.widgets.size <= 1) {
                    "Given card should have only one widget"
                }
                val widget = card.widgets[0]
                check(WIDGET_TYPE_IMAGE == widget.type) { "Widget type should be image." }
                imageUrls.add(widget.content)
            }
            return imageUrls
        }

    private fun getCarouselRemoteView(isAutoStart: Boolean, isPersistent: Boolean): RemoteViews {
        return if (doesSdkSupportDecoratedStyleOnDevice()) {
            if (isPersistent) {
                RemoteViews(
                    context.packageName,
                    R.layout.moe_rich_push_simple_carousel_decorated_style_expanded_view_with_dismiss_cta_layout
                )
            } else {
                RemoteViews(
                    context.packageName,
                    R.layout.moe_rich_push_simple_carousel_decorated_style_expanded_view_without_dismiss_cta_layout
                )
            }
        } else {
            if (isAutoStart) {
                RemoteViews(
                    context.packageName,
                    getTemplateLayout(
                        R.layout.moe_rich_push_simple_carousel_auto_start_expanded_view,
                        R.layout.moe_rich_push_simple_carousel_auto_start_expanded_view_layout_big,
                        sdkInstance
                    )
                )
            } else {
                RemoteViews(
                    context.packageName,
                    getTemplateLayout(
                        R.layout.moe_rich_push_simple_carousel_manual_expanded_view,
                        R.layout.moe_rich_push_simple_carousel_manual_expanded_view_layout_big,
                        sdkInstance
                    )
                )
            }
        }
    }

    @Throws(JSONException::class)
    private fun removeFailedImagesFromPayload() {
        sdkInstance.logger.log { "$tag removeFailedImagesFromPayload() : Will remove failed images from payload." }
        val featuresJson =
            JSONObject(metaData.payload.payload.getString(PAYLOAD_ATTRIBUTE_MOE_FEATURES) ?: return)
        val richPush =
            featuresJson.getJSONObject(PAYLOAD_ATTRIBUTE_RICH_PUSH)
        val expandedJson =
            richPush.getJSONObject(EXPANDED_CUSTOMISATION)
        val cardsJson = expandedJson.getJSONArray(CARDS)
        val downloadedCardsJson = JSONArray()
        val imageManager =
            ImageManager(context, sdkInstance)
        val updatedCardList = mutableListOf<Card>()
        for (index in template.expandedTemplate!!.cards.indices) {
            val card =
                template.expandedTemplate.cards[index]
            if (imageManager.isImageExist(
                    metaData.payload.campaignId,
                    card.widgets[0].content
                )
            ) {
                downloadedCardsJson.put(cardsJson.getJSONObject(index))
                updatedCardList.add(card)
            } else {
                sdkInstance.logger.log { "$tag removeFailedImagesFromPayload() : Removing card as image download failed. Index: $index" }
            }
        }
        template.expandedTemplate.cards = updatedCardList
        expandedJson.put(CARDS, downloadedCardsJson)
        richPush.put(EXPANDED_CUSTOMISATION, expandedJson)
        sdkInstance.logger.log { "$tag removeFailedImagesFromPayload() : Updated Rich push payload: $richPush" }
        featuresJson.put(PAYLOAD_ATTRIBUTE_RICH_PUSH, richPush)
        metaData.payload.payload.putString(PAYLOAD_ATTRIBUTE_MOE_FEATURES, featuresJson.toString())
    }

    private fun getDirectionIntent(
        context: Context,
        payloadBundle: Bundle,
        notificationId: Int,
        navigationDirection: String,
        imageIndex: Int,
        totalImages: Int
    ): Intent {
        val directionIntent =
            Intent(context, MoERichPushIntentService::class.java)
        directionIntent.flags = Intent.FLAG_ACTIVITY_NEW_TASK
        directionIntent.putExtras(payloadBundle)
            .putExtra(NAVIGATION_DIRECTION, navigationDirection)
            .putExtra(IMAGE_INDEX, imageIndex)
            .putExtra(TOTAL_IMAGE_COUNT, totalImages)
            .putExtra(MOE_NOTIFICATION_ID, notificationId)
        return directionIntent
    }

    private fun downloadAndSaveImages(imageUrls: List<String>): Int {
        val successCount = intArrayOf(0)
        try {
            sdkInstance.logger.log { "$tag downloadAndSaveImages() : Downloading images for template." }
            val service = Executors.newCachedThreadPool()
            val fileManager = ImageManager(context, sdkInstance)
            for (imageUrl in imageUrls) {
                service.submit {
                    try {
                        sdkInstance.logger.log { "$tag run() : Will try to download image: $imageUrl" }
                        val bitmap = downloadImageBitmap(imageUrl)
                        if (bitmap != null && fileManager.saveImage(
                                metaData.payload.campaignId,
                                imageUrl,
                                bitmap
                            )
                        ) {
                            sdkInstance.logger.log { "$tag run() : Successfully downloaded image:$imageUrl" }
                            successCount[0]++
                        }
                    } catch (t: Throwable) {
                        sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag run() : " }
                    }
                }
            }
            service.shutdown()
            service.awaitTermination(10, TimeUnit.SECONDS)
            sdkInstance.logger.log { "$tag downloadAndSaveImages() : Download complete, success count: " + successCount[0] }
        } catch (e: InterruptedException) {
            sdkInstance.logger.log(LogLevel.ERROR, e) { "$tag downloadAndSaveImages() : " }
        }
        return successCount[0]
    }

    private val singleImage =
        arrayOf(
            CardWidget(
                cardId = R.id.card11,
                verticalImageId = R.id.verticalImage11,
                horizontalCenterCropImageId = R.id.horizontalCenterCropImage11,
                horizontalFitCenterImageId = R.id.horizontalCenterCropImage11
            )
        )
    private val viewFlipperTwoImageIdArray = arrayOf(
        CardWidget(
            cardId = R.id.card21,
            verticalImageId = R.id.verticalImage21,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage21,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage21
        ),
        CardWidget(
            cardId = R.id.card22,
            verticalImageId = R.id.verticalImage22,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage22,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage22
        )
    )
    private val viewFlipperThreeImageIdArray = arrayOf(
        CardWidget(
            cardId = R.id.card31,
            verticalImageId = R.id.verticalImage31,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage31,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage31
        ),
        CardWidget(
            cardId = R.id.card32,
            verticalImageId = R.id.verticalImage32,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage32,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage32
        ),
        CardWidget(
            cardId = R.id.card33,
            verticalImageId = R.id.verticalImage33,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage33,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage33
        )
    )
    private val viewFlipperFourImageIdArray = arrayOf(
        CardWidget(
            cardId = R.id.card41,
            verticalImageId = R.id.verticalImage41,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage41,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage41
        ),
        CardWidget(
            cardId = R.id.card42,
            verticalImageId = R.id.verticalImage42,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage42,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage42
        ),
        CardWidget(
            cardId = R.id.card43,
            verticalImageId = R.id.verticalImage43,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage43,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage43
        ),
        CardWidget(
            cardId = R.id.card44,
            verticalImageId = R.id.verticalImage44,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage44,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage44
        )
    )
    private val viewFlipperFiveImageIdArray = arrayOf(
        CardWidget(
            cardId = R.id.card51,
            verticalImageId = R.id.verticalImage51,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage51,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage51
        ),
        CardWidget(
            cardId = R.id.card52,
            verticalImageId = R.id.verticalImage52,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage52,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage52
        ),
        CardWidget(
            cardId = R.id.card53,
            verticalImageId = R.id.verticalImage53,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage53,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage53
        ),
        CardWidget(
            cardId = R.id.card54,
            verticalImageId = R.id.verticalImage54,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage54,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage54
        ),
        CardWidget(
            cardId = R.id.card55,
            verticalImageId = R.id.verticalImage55,
            horizontalCenterCropImageId = R.id.horizontalCenterCropImage55,
            horizontalFitCenterImageId = R.id.horizontalFitCenterImage55
        )
    )
    private val markerImageIdArray =
        intArrayOf(R.id.marker1, R.id.marker2, R.id.marker3, R.id.marker4, R.id.marker5)
}