/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.richnotification.internal.builder

import android.content.Context
import android.os.Build
import android.view.View
import android.widget.ImageView.ScaleType
import android.widget.RemoteViews
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.downloadImageBitmap
import com.moengage.pushbase.internal.model.NotificationMetaData
import com.moengage.richnotification.R
import com.moengage.richnotification.internal.ASSET_COLOR_DARK_GREY
import com.moengage.richnotification.internal.Evaluator
import com.moengage.richnotification.internal.MAX_IMAGE_BANNER_HEIGHT_BIG_LAYOUT_COLLAPSED_STATE
import com.moengage.richnotification.internal.MAX_IMAGE_BANNER_HEIGHT_BIG_LAYOUT_EXPANDED_STATE
import com.moengage.richnotification.internal.MAX_IMAGE_BANNER_HEIGHT_COLLAPSED_STATE
import com.moengage.richnotification.internal.MAX_IMAGE_BANNER_HEIGHT_EXPANDED_STATE
import com.moengage.richnotification.internal.MODULE_TAG
import com.moengage.richnotification.internal.WIDGET_TYPE_IMAGE
import com.moengage.richnotification.internal.WIDGET_TYPE_TEXT
import com.moengage.richnotification.internal.doesSdkSupportDecoratedStyleOnDevice
import com.moengage.richnotification.internal.getAppName
import com.moengage.richnotification.internal.getDecoratedStyleTemplateLayout
import com.moengage.richnotification.internal.getHtmlText
import com.moengage.richnotification.internal.getTemplateLayout
import com.moengage.richnotification.internal.getTime
import com.moengage.richnotification.internal.isBigLayoutSupported
import com.moengage.richnotification.internal.models.Card
import com.moengage.richnotification.internal.models.CollapsedBannerTemplate
import com.moengage.richnotification.internal.models.ExpandedBannerTemplate
import com.moengage.richnotification.internal.models.HeaderStyle
import com.moengage.richnotification.internal.models.ImageWidget
import com.moengage.richnotification.internal.models.Template

/**
 * @author Umang Chamaria
 */
internal class ImageBannerBuilder(
    private val context: Context,
    private val template: Template,
    private val metaData: NotificationMetaData,
    private val sdkInstance: SdkInstance
) {

    private val tag = "${MODULE_TAG}ImageBannerBuilder"

    fun buildCollapsedImageBanner(): Boolean {
        try {
            sdkInstance.logger.log { "$tag buildCollapsedImageBanner() : Will try to build image banner template" }
            if (template.collapsedTemplate == null || template.collapsedTemplate !is CollapsedBannerTemplate) return false
            val collapsedTemplate = template.collapsedTemplate
            sdkInstance.logger.log { "$tag buildCollapsedImageBanner() : Collapsed template: $collapsedTemplate" }
            val remoteViews = getCollapsedImageBannerLayout()
            if (collapsedTemplate.cards.isEmpty()) return false
            val templateHelper = TemplateHelper(sdkInstance)
            // layout style
            templateHelper.addLayoutStyle(
                collapsedTemplate.layoutStyle,
                remoteViews,
                R.id.collapsedRootView
            )
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                // Requires to remove the summary which was set during building basic template, ImageBanner doesn't support summary
                metaData.notificationBuilder.setSubText("")
            } else {
                addHeaderWithPersistentAssets(
                    templateHelper,
                    remoteViews,
                    collapsedTemplate.isHeaderEnabled
                )
            }
            val card = collapsedTemplate.cards[0]
            if (card.widgets.isEmpty()) return false
            val widget = card.widgets[0]
            if (WIDGET_TYPE_IMAGE != widget.type) return false
            if (!templateHelper.addImageWidgetToTemplate(
                    context,
                    metaData,
                    template,
                    remoteViews,
                    widget as ImageWidget,
                    card,
                    maxAllowedImageHeight = getMaxAllowedImageHeightForImageBanner(true)
                )
            ) {
                return false
            }
            templateHelper.addDefaultActionToNotificationClick(
                context,
                remoteViews,
                R.id.collapsedRootView,
                template,
                metaData
            )
            metaData.notificationBuilder.setCustomContentView(remoteViews)
            return true
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag buildCollapsedImageBanner() : " }
        }
        return false
    }

    private fun getCollapsedImageBannerLayout(): RemoteViews {
        return if (doesSdkSupportDecoratedStyleOnDevice()) {
            RemoteViews(
                context.packageName,
                getDecoratedStyleTemplateLayout(
                    R.layout.moe_rich_push_image_banner_collapsed_small_layout_decorated_style,
                    R.layout.moe_rich_push_image_banner_collapsed_layout_decorated_style,
                    sdkInstance
                )
            )
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            RemoteViews(
                context.packageName,
                getTemplateLayout(
                    layoutSmall = R.layout.moe_rich_push_image_banner_collapsed,
                    layoutBig = R.layout.moe_rich_push_image_banner_collapsed_layout_big,
                    sdkInstance
                )
            )
        } else {
            RemoteViews(
                context.packageName,
                R.layout.moe_rich_push_image_banner_collapsed_below_m
            )
        }
    }

    fun buildExpandedImageBanner(): Boolean {
        try {
            sdkInstance.logger.log { "$tag buildExpandedImageBanner() : Will try to build image banner." }
            if (template.expandedTemplate == null || template.expandedTemplate !is ExpandedBannerTemplate) return false
            val expandedTemplate = template.expandedTemplate
            sdkInstance.logger.log { "$tag buildExpandedImageBanner() : Template: $expandedTemplate" }
            if (expandedTemplate.cards.isEmpty()) return false
            val remoteViews = getExpandedImageBannerLayout(metaData.payload.addOnFeatures.isPersistent)
            val templateHelper = TemplateHelper(sdkInstance)
            // layout style
            templateHelper.addLayoutStyle(
                expandedTemplate.layoutStyle,
                remoteViews,
                R.id.expandedRootView
            )
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                // Requires to remove the summary which was set during building basic template, ImageBanner doesn't support summary
                metaData.notificationBuilder.setSubText("")
                if (metaData.payload.addOnFeatures.isPersistent) {
                    templateHelper.setDismissCtaCustomization(remoteViews, template.dismissCta)
                    templateHelper.addActionToDismissCTA(remoteViews, context, metaData)
                }
            } else {
                addHeaderWithPersistentAssets(
                    templateHelper,
                    remoteViews,
                    expandedTemplate.isHeaderEnabled
                )
            }

            val card = expandedTemplate.cards[0]
            if (card.widgets.isEmpty()) return false
            val widget = card.widgets[0]
            if (WIDGET_TYPE_IMAGE != widget.type) return false
            if (!templateHelper.addImageWidgetToTemplate(
                    context,
                    metaData,
                    template,
                    remoteViews,
                    widget as ImageWidget,
                    card,
                    maxAllowedImageHeight = getMaxAllowedImageHeightForImageBanner(false)
                )
            ) {
                return false
            }
            templateHelper.addDefaultActionToNotificationClick(
                context,
                remoteViews,
                R.id.expandedRootView,
                template,
                metaData
            )
            metaData.notificationBuilder.setCustomBigContentView(remoteViews)
            return true
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag buildExpandedImageBanner() : " }
        }
        return false
    }

    private fun getExpandedImageBannerLayout(isPersistent: Boolean): RemoteViews {
        return if (doesSdkSupportDecoratedStyleOnDevice()) {
            if (isPersistent) {
                RemoteViews(
                    context.packageName,
                    R.layout.moe_rich_push_image_banner_expanded_layout_decorated_style_with_dismiss
                )
            } else {
                RemoteViews(
                    context.packageName,
                    R.layout.moe_rich_push_image_banner_expanded_layout_decorated_style
                )
            }
        } else {
            RemoteViews(
                context.packageName,
                getTemplateLayout(
                    layoutSmall = R.layout.moe_rich_push_image_banner_expanded,
                    layoutBig = R.layout.moe_rich_push_image_banner_expanded_layout_big,
                    sdkInstance
                )
            )
        }
    }

    fun buildExpandedImageBannerText(): Boolean {
        try {
            sdkInstance.logger.log { "$tag buildExpandedImageBannerText() : Will try to build image banner text." }
            if (template.expandedTemplate == null || template.expandedTemplate !is ExpandedBannerTemplate) return false
            val expandedTemplate = template.expandedTemplate
            sdkInstance.logger.log { "$tag buildExpandedImageBannerText() : Template payload: $expandedTemplate" }
            if (expandedTemplate.cards.isEmpty()) return false
            val card = expandedTemplate.cards[0]
            if (!Evaluator(sdkInstance.logger).isValidBannerTextTemplate(card)) return false
            val remoteViews = getExpandedImageBannerTextLayout(metaData.payload.addOnFeatures.isPersistent)
            val templateHelper = TemplateHelper(sdkInstance)
            // layout style
            templateHelper.addLayoutStyle(
                expandedTemplate.layoutStyle,
                remoteViews,
                R.id.expandedRootView
            )
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                // Requires to remove the summary which was set during building basic template, ImageBanner doesn't support summary
                metaData.notificationBuilder.setSubText("")
                if (metaData.payload.addOnFeatures.isPersistent) {
                    templateHelper.setDismissCtaCustomization(remoteViews, template.dismissCta)
                    templateHelper.addActionToDismissCTA(remoteViews, context, metaData)
                }
            } else {
                addHeaderWithPersistentAssets(
                    templateHelper,
                    remoteViews,
                    expandedTemplate.isHeaderEnabled
                )
            }

            for (widget in card.widgets) {
                if (widget.id == 0 && WIDGET_TYPE_IMAGE == widget.type) {
                    if (!addImageToTextOverlayBannerTemplate(
                            context,
                            metaData,
                            template,
                            templateHelper,
                            remoteViews,
                            widget as ImageWidget,
                            card
                        )
                    ) {
                        return false
                    }
                } else if (widget.id == 1 && WIDGET_TYPE_TEXT == widget.type) {
                    if (widget.content.isBlank()) continue
                    remoteViews.setTextViewText(
                        R.id.headerText,
                        getHtmlText(widget.content)
                    )
                    remoteViews.setViewVisibility(R.id.headerText, View.VISIBLE)
                } else if (widget.id == 2 && WIDGET_TYPE_TEXT == widget.type) {
                    if (widget.content.isBlank()) continue
                    remoteViews.setTextViewText(
                        R.id.messageText,
                        getHtmlText(widget.content)
                    )
                    remoteViews.setViewVisibility(R.id.messageText, View.VISIBLE)
                } else {
                    sdkInstance.logger.log(LogLevel.WARN) { "$tag buildExpandedImageBannerText() : Unknown widget. Ignoring" }
                }
            }
            templateHelper.addDefaultActionToNotificationClick(
                context,
                remoteViews,
                R.id.expandedRootView,
                template,
                metaData
            )
            metaData.notificationBuilder.setCustomBigContentView(remoteViews)
            return true
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag buildExpandedImageBannerText() : " }
        }
        return false
    }

    private fun getExpandedImageBannerTextLayout(isPersistent: Boolean): RemoteViews {
        return if (doesSdkSupportDecoratedStyleOnDevice()) {
            if (isPersistent) {
                RemoteViews(
                    context.packageName,
                    R.layout.moe_rich_push_image_banner_text_expanded_layout_decorated_style_with_dismiss
                )
            } else {
                RemoteViews(
                    context.packageName,
                    R.layout.moe_rich_push_image_banner_text_expanded_layout_decorated_style
                )
            }
        } else {
            RemoteViews(
                context.packageName,
                getTemplateLayout(
                    layoutSmall = R.layout.moe_rich_push_image_banner_text_expanded,
                    layoutBig = R.layout.moe_rich_push_image_banner_text_expanded_layout_big,
                    sdkInstance
                )
            )
        }
    }

    private fun addHeader(
        remoteViews: RemoteViews,
        isHeaderEnabled: Boolean,
        headerStyle: HeaderStyle
    ) {
        if (!isHeaderEnabled) return
        remoteViews.setViewVisibility(R.id.appInfo, View.VISIBLE)
        remoteViews.setImageViewResource(R.id.smallIcon, sdkInstance.initConfig.push.meta.smallIcon)
        val templateHelper = TemplateHelper(sdkInstance)
        templateHelper.setSmallIconColor(context, remoteViews)
        remoteViews.setTextViewText(R.id.time, getTime())
        remoteViews.setTextViewText(R.id.appName, getAppName(context))
        templateHelper.setHeaderStyle(remoteViews, headerStyle)
        remoteViews.setImageViewResource(
            R.id.separatorTime,
            if (template.assetColor == ASSET_COLOR_DARK_GREY) {
                R.drawable.moe_rich_push_dark_separator
            } else {
                R.drawable.moe_rich_push_light_separator
            }
        )
    }

    private fun addImageToTextOverlayBannerTemplate(
        context: Context,
        metaData: NotificationMetaData,
        template: Template,
        templateHelper: TemplateHelper,
        remoteView: RemoteViews,
        widget: ImageWidget,
        card: Card
    ): Boolean {
        val bitmap = downloadImageBitmap(widget.content) ?: return false
        val widgetId: Int = if (doesSdkSupportDecoratedStyleOnDevice()) {
            if (widget.scaleType == ScaleType.CENTER_CROP) {
                templateHelper.setViewCornerToRounded(remoteView, R.id.centerCropImage)
                R.id.centerCropImage
            } else {
                R.id.centerInsideImage
            }
        } else {
            R.id.imageBanner
        }
        remoteView.setImageViewBitmap(widgetId, bitmap)
        remoteView.setViewVisibility(widgetId, View.VISIBLE)
        templateHelper.addActionToImageWidget(
            context,
            metaData,
            template,
            remoteView,
            widget,
            card,
            widgetId
        )
        return true
    }

    private fun addHeaderWithPersistentAssets(
        templateHelper: TemplateHelper,
        remoteViews: RemoteViews,
        isHeaderEnabled: Boolean
    ) {
        if (metaData.payload.addOnFeatures.isPersistent) {
            templateHelper.addPersistenceAsset(
                template.assetColor,
                remoteViews,
                R.id.closeButton
            )
            templateHelper.addActionToDismissCTA(remoteViews, context, metaData)
        }
        addHeader(remoteViews, isHeaderEnabled, template.headerStyle)
    }

    private fun getMaxAllowedImageHeightForImageBanner(
        isCollapsedState: Boolean
    ): Int {
        return if (isCollapsedState) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M && isBigLayoutSupported(sdkInstance.remoteConfig)) {
                MAX_IMAGE_BANNER_HEIGHT_BIG_LAYOUT_COLLAPSED_STATE
            } else {
                MAX_IMAGE_BANNER_HEIGHT_COLLAPSED_STATE
            }
        } else {
            if (isBigLayoutSupported(sdkInstance.remoteConfig)) {
                MAX_IMAGE_BANNER_HEIGHT_BIG_LAYOUT_EXPANDED_STATE
            } else {
                MAX_IMAGE_BANNER_HEIGHT_EXPANDED_STATE
            }
        }
    }
}