package com.moengage.richnotification.internal.builder

import android.content.Context
import android.os.Build
import android.widget.RemoteViews
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.pushbase.internal.model.NotificationMetaData
import com.moengage.richnotification.R
import com.moengage.richnotification.internal.DECORATED_STYLE_MAX_LINES_WITHOUT_ACTION_IMAGE
import com.moengage.richnotification.internal.DECORATED_STYLE_MAX_LINES_WITH_ACTION_IMAGE
import com.moengage.richnotification.internal.DECORATED_STYLE_MAX_LINES_WITH_ACTION_WITHOUT_IMAGE
import com.moengage.richnotification.internal.Evaluator
import com.moengage.richnotification.internal.MODULE_TAG
import com.moengage.richnotification.internal.doesSdkSupportDecoratedStyleOnDevice
import com.moengage.richnotification.internal.getAppName
import com.moengage.richnotification.internal.getDecoratedStyleTemplateLayout
import com.moengage.richnotification.internal.getTemplateLayout
import com.moengage.richnotification.internal.models.Template

/**
 * @author Abhishek Kumar
 * Date: 12/10/2022
 */
internal class StylizedBasicTemplateBuilder(
    private val context: Context,
    private val template: Template,
    private val metaData: NotificationMetaData,
    private val sdkInstance: SdkInstance
) {

    private val tag = "${MODULE_TAG}StylizedBasicTemplateBuilder"

    fun buildCollapsedStylizedBasic(): Boolean {
        try {
            sdkInstance.logger.log { "$tag buildCollapsedStylizedBasic() : Will try to build collapsed stylised basic template" }
            if (!Evaluator(sdkInstance.logger).hasMinimumText(template.defaultText)) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag buildCollapsedStylizedBasic() : Does not have minimum text." }
                return false
            }
            if (template.collapsedTemplate == null) return false
            val remoteViews = getCollapsedStylizedBasicRemoteView()
            val templateHelper = TemplateHelper(sdkInstance)
            templateHelper.addLayoutStyle(
                template.collapsedTemplate.layoutStyle,
                remoteViews,
                R.id.collapsedRootView
            )
            // default text
            templateHelper.setDefaultTextAndStyle(
                remoteViews,
                template.defaultText,
                getAppName(context),
                template.headerStyle
            )
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                templateHelper.addDecoratedStyleBaseProperties(
                    remoteViews,
                    R.id.collapsedRootView,
                    template,
                    metaData
                )
            } else {
                templateHelper.setHeaderAssetsAndIcon(context, remoteViews, template, metaData)
                // persistent notification
                if (metaData.payload.addOnFeatures.isPersistent) {
                    templateHelper.addActionToDismissCTA(remoteViews, context, metaData)
                }
            }
            // large icon
            templateHelper.addLargeIcon(remoteViews, template, metaData.payload)
            templateHelper.addDefaultActionToNotificationClick(
                context,
                remoteViews,
                R.id.collapsedRootView,
                template,
                metaData
            )
            metaData.notificationBuilder.setCustomContentView(remoteViews)
            return true
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag buildCollapsedStylizedBasic() : " }
        }
        return false
    }

    private fun getCollapsedStylizedBasicRemoteView(): RemoteViews {
        return if (doesSdkSupportDecoratedStyleOnDevice()) {
            RemoteViews(
                context.packageName,
                getDecoratedStyleTemplateLayout(
                    R.layout.moe_rich_push_stylized_basic_collapsed_small_layout_decorated_style,
                    R.layout.moe_rich_push_stylized_basic_collapsed_layout_decorated_style,
                    sdkInstance
                )
            )
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            RemoteViews(
                context.packageName,
                getTemplateLayout(
                    R.layout.moe_rich_push_stylized_basic_collapsed,
                    R.layout.moe_rich_push_stylized_basic_collapsed_layout_big,
                    sdkInstance
                )
            )
        } else {
            RemoteViews(
                context.packageName,
                R.layout.moe_rich_push_stylized_basic_collapsed_below_m
            )
        }
    }

    fun buildExpandedStylizedBasic(): Boolean {
        try {
            if (template.expandedTemplate == null) return false
            if (!Evaluator(sdkInstance.logger).hasMinimumText(template.defaultText)) {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag buildExpandedStylizedBasic() : Does not have minimum text." }
                return false
            }
            sdkInstance.logger.log { "$tag buildExpandedStylizedBasic() : Will build stylized basic template." }
            sdkInstance.logger.log { "$tag buildExpandedStylizedBasic() : Template: ${template.expandedTemplate}" }
            val remoteViews = getExpandedStylizedBasicRemoteView(
                template.expandedTemplate.actionButtonList.isNotEmpty(),
                metaData.payload.addOnFeatures.isPersistent
            )
            // BigTextStyle
            if (template.expandedTemplate.cards.isEmpty() &&
                template.expandedTemplate.actionButtonList.isEmpty() &&
                !(doesSdkSupportDecoratedStyleOnDevice() && metaData.payload.addOnFeatures.isPersistent)
            ) {
                return false
            }

            val templateHelper = TemplateHelper(sdkInstance)
            templateHelper.addLayoutStyle(
                template.expandedTemplate.layoutStyle,
                remoteViews,
                R.id.expandedRootView
            )
            // default text
            templateHelper.setDefaultTextAndStyle(
                remoteViews,
                template.defaultText,
                getAppName(context),
                template.headerStyle
            )
            var isImageAddedToExpandedTemplate = false
            if (template.expandedTemplate.cards.isNotEmpty()) {
                isImageAddedToExpandedTemplate = templateHelper.addImageToExpandedTemplate(
                    context,
                    metaData,
                    template,
                    remoteViews
                )
            } else {
                templateHelper.removeImageWidgetFromExpandedTemplate(remoteViews)
            }
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                addDecoratedStyleExpandedProperties(
                    metaData.payload.addOnFeatures.isPersistent,
                    template,
                    remoteViews,
                    templateHelper,
                    isImageAddedToExpandedTemplate
                )
            } else {
                addNonDecoratedStyleExpandedProperties(
                    template,
                    remoteViews,
                    templateHelper,
                    isImageAddedToExpandedTemplate
                )
            }
            templateHelper.addLargeIcon(remoteViews, template, metaData.payload)
            if (template.expandedTemplate.actionButtonList.isNotEmpty() ||
                metaData.payload.addOnFeatures.isPersistent
            ) {
                templateHelper.addActionButton(
                    context,
                    metaData,
                    template,
                    remoteViews,
                    template.expandedTemplate.actionButtonList,
                    metaData.payload.addOnFeatures.isPersistent
                )
            }
            templateHelper.addDefaultActionToNotificationClick(
                context,
                remoteViews,
                R.id.collapsedRootView,
                template,
                metaData
            )
            metaData.notificationBuilder.setCustomBigContentView(remoteViews)
            return true
        } catch (t: Throwable) {
            sdkInstance.logger.log(
                LogLevel.ERROR,
                t
            ) { "$tag buildExpandedStylizedBasic() : Exception " }
        }
        return false
    }

    private fun getExpandedStylizedBasicRemoteView(
        hasActionButton: Boolean,
        isPersistent: Boolean
    ): RemoteViews {
        return if (doesSdkSupportDecoratedStyleOnDevice()) {
            if (hasActionButton || isPersistent) {
                // Android 12+ With Cta (Dismiss button act as cta in Android 12+)
                RemoteViews(
                    context.packageName,
                    R.layout.moe_rich_push_stylized_basic_big_picture_with_action_button_decorated_style
                )
            } else {
                // Android 12+ Without Cta
                RemoteViews(
                    context.packageName,
                    R.layout.moe_rich_push_stylized_basic_big_picture_without_action_button_decorated_style
                )
            }
        } else {
            if (hasActionButton) {
                // Below Android 12 With Cta
                RemoteViews(
                    context.packageName,
                    getTemplateLayout(
                        R.layout.moe_rich_push_stylized_basic_big_picture_with_action_button,
                        R.layout.moe_rich_push_stylized_basic_big_picture_with_action_button_layout_big,
                        sdkInstance
                    )
                )
            } else {
                // Below Android 12 Without Cta
                RemoteViews(
                    context.packageName,
                    getTemplateLayout(
                        R.layout.moe_rich_push_stylized_basic_big_picture_without_action_button,
                        R.layout.moe_rich_push_stylized_basic_big_picture_without_action_button_layout_big,
                        sdkInstance
                    )
                )
            }
        }
    }

    private fun addDecoratedStyleExpandedProperties(
        isPersistent: Boolean,
        template: Template,
        remoteViews: RemoteViews,
        templateHelper: TemplateHelper,
        isImageAddedToExpandedTemplate: Boolean
    ) {
        if (template.expandedTemplate == null) return
        if (template.expandedTemplate.cards.isNotEmpty() && isImageAddedToExpandedTemplate) {
            // With Image
            remoteViews.setInt(
                R.id.message,
                "setMaxLines",
                DECORATED_STYLE_MAX_LINES_WITH_ACTION_IMAGE
            )
        } else if (template.expandedTemplate.actionButtonList.isNotEmpty() || isPersistent) {
            // Without Image + With Action Button
            remoteViews.setInt(
                R.id.message,
                "setMaxLines",
                DECORATED_STYLE_MAX_LINES_WITH_ACTION_WITHOUT_IMAGE
            )
        } else {
            // Without Image & Action Button
            remoteViews.setInt(
                R.id.message,
                "setMaxLines",
                DECORATED_STYLE_MAX_LINES_WITHOUT_ACTION_IMAGE
            )
        }
        templateHelper.addDecoratedStyleBaseProperties(
            remoteViews,
            R.id.expandedRootView,
            template,
            metaData
        )
    }

    private fun addNonDecoratedStyleExpandedProperties(
        template: Template,
        remoteViews: RemoteViews,
        templateHelper: TemplateHelper,
        isImageAddedToExpandedTemplate: Boolean
    ) {
        if (template.expandedTemplate == null) return
        if (template.expandedTemplate.cards.isNotEmpty() && isImageAddedToExpandedTemplate) {
            // With Image
            remoteViews.setBoolean(R.id.message, "setSingleLine", true)
            remoteViews.setInt(R.id.message, "setMaxLines", 1)
        } else if (template.expandedTemplate.actionButtonList.isNotEmpty()) {
            // Without Image + With Action Button
            remoteViews.setBoolean(R.id.message, "setSingleLine", false)
            remoteViews.setInt(R.id.message, "setMaxLines", 10)
        } else {
            // Without Image & Action Button
            remoteViews.setBoolean(R.id.message, "setSingleLine", false)
            remoteViews.setInt(R.id.message, "setMaxLines", 13)
        }
        templateHelper.setHeaderAssetsAndIcon(
            context,
            remoteViews,
            template,
            metaData
        )
    }
}