/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.richnotification.internal.builder

import android.content.Context
import android.widget.RemoteViews
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.logBundle
import com.moengage.pushbase.internal.PAYLOAD_ATTRIBUTE_MOE_FEATURES
import com.moengage.pushbase.internal.TEMPLATE_META
import com.moengage.pushbase.internal.model.NotificationMetaData
import com.moengage.pushbase.internal.model.RichPushTemplateState
import com.moengage.pushbase.internal.model.TemplateTrackingMeta
import com.moengage.pushbase.internal.templateTrackingMetaToJsonString
import com.moengage.richnotification.R
import com.moengage.richnotification.internal.DECORATED_STYLE_MAX_LINES_WITHOUT_ACTION_IMAGE
import com.moengage.richnotification.internal.DECORATED_STYLE_MAX_LINES_WITH_ACTION_WITHOUT_IMAGE
import com.moengage.richnotification.internal.Evaluator
import com.moengage.richnotification.internal.MODULE_TAG
import com.moengage.richnotification.internal.PROGRESS_ALARM_ID
import com.moengage.richnotification.internal.TEMPLATE_NAME_IMAGE_BANNER
import com.moengage.richnotification.internal.TEMPLATE_NAME_TIMER_WITH_PROGRESS
import com.moengage.richnotification.internal.TIMER_ALARM_ID
import com.moengage.richnotification.internal.addProgressPropertiesIfRequired
import com.moengage.richnotification.internal.doesSdkSupportDecoratedStyleOnDevice
import com.moengage.richnotification.internal.getAppName
import com.moengage.richnotification.internal.getNotificationClearIntent
import com.moengage.richnotification.internal.getProgressUpdateAlarmId
import com.moengage.richnotification.internal.getTemplateLayout
import com.moengage.richnotification.internal.getTimerEndTime
import com.moengage.richnotification.internal.getTimerExpiryAlarmId
import com.moengage.richnotification.internal.isPushTemplateSupported
import com.moengage.richnotification.internal.models.ProgressProperties
import com.moengage.richnotification.internal.models.Template
import com.moengage.richnotification.internal.models.TimerTemplate
import com.moengage.richnotification.internal.repository.PayloadParser
import com.moengage.richnotification.internal.setNotificationClearIntent
import com.moengage.richnotification.internal.setUpTimerAndProgressComponents
import com.moengage.richnotification.internal.updateDefaultAction

/**
 * @author Umang Chamaria
 * Date: 05/04/20
 */
internal class TemplateBuilder {

    private val tag = "${MODULE_TAG}TemplateBuilder"

    fun buildTemplate(
        context: Context,
        metaData: NotificationMetaData,
        sdkInstance: SdkInstance
    ): RichPushTemplateState {
        try {
            sdkInstance.logger.log { "$tag buildTemplate() : Will attempt to build template." }
            if (!isPushTemplateSupported(metaData.payload, sdkInstance)) return RichPushTemplateState()
            // get the rich push payload string
            val template = metaData.payload.payload.getString(PAYLOAD_ATTRIBUTE_MOE_FEATURES)
                ?.let { PayloadParser().parseTemplate(it) } ?: return RichPushTemplateState()
            sdkInstance.logger.log { "$tag buildTemplate() : Template: $template" }
            updateDefaultAction(template, metaData, sdkInstance)
            metaData.payload.payload.putString(
                TEMPLATE_META,
                templateTrackingMetaToJsonString(
                    TemplateTrackingMeta(
                        template.templateName,
                        -1,
                        -1
                    )
                )
            )
            sdkInstance.logger.log { "$tag buildTemplate() : Updated payload: " }
            logBundle(sdkInstance.logger, tag, metaData.payload.payload)
            val isTimerTemplate =
                Evaluator(sdkInstance.logger).isTimerTemplate(
                    template.collapsedTemplate?.type,
                    template.expandedTemplate?.type
                ) && template is TimerTemplate

            var progressProperties = getTimerEndTime(template)

            if (!shouldBuildTemplate(
                    template,
                    sdkInstance,
                    progressProperties
                )
            ) {
                return RichPushTemplateState(shouldReRenderBackupTemplate = false)
            }

            progressProperties =
                setUpProgressPropertiesIfRequired(
                    isTimerTemplate,
                    template,
                    metaData,
                    sdkInstance,
                    progressProperties
                )

            val hasCustomCollapsed = CollapsedTemplateBuilder(
                context,
                template,
                metaData,
                sdkInstance,
                progressProperties
            ).build()

            var hasCustomExpanded = ExpandedTemplateBuilder(
                context,
                template,
                metaData,
                sdkInstance,
                progressProperties
            ).build()

            if (!hasCustomCollapsed && !hasCustomExpanded) return RichPushTemplateState()

            if (shouldAddBigText(template, hasCustomCollapsed, hasCustomExpanded)) {
                sdkInstance.logger.log { "$tag buildTemplate() : Will add big text to notification" }
                hasCustomExpanded =
                    buildBigTextStyleNotification(context, template, metaData, sdkInstance)
            }
            val hasTemplate = hasCustomCollapsed || hasCustomExpanded

            if (hasTemplate) {
                if (isTimerTemplate) {
                    progressProperties.timerEndTime = getTimerEndTime(
                        progressProperties.timerProperties.duration,
                        progressProperties.timerProperties.expiry
                    )
                    setUpTimerAndProgressComponents(
                        context,
                        template,
                        metaData,
                        sdkInstance,
                        progressProperties
                    )
                }
                updateNotificationClearCallback(context, template, metaData)
            }
            return RichPushTemplateState(
                hasCustomCollapsedTemplate = hasCustomCollapsed,
                hasCustomExpandedTemplate = hasCustomExpanded
            )
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag buildTemplate() : " }
        }
        return RichPushTemplateState()
    }

    private fun buildBigTextStyleNotification(
        context: Context,
        template: Template,
        metaData: NotificationMetaData,
        sdkInstance: SdkInstance
    ): Boolean {
        try {
            sdkInstance.logger.log { "$tag buildBigTextStyleNotification() : Building big text notification. $template" }
            val remoteViews = getBigTextStyleRemoteView(context, sdkInstance)
            val templateHelper = TemplateHelper(sdkInstance)
            templateHelper.addLayoutStyle(
                template.collapsedTemplate?.layoutStyle,
                remoteViews,
                R.id.expandedRootView
            )
            remoteViews.setInt(
                R.id.message,
                "setMaxLines",
                getMaxMessageLinesToShow(
                    doesSdkSupportDecoratedStyleOnDevice(),
                    metaData.payload.addOnFeatures.isPersistent
                )
            )
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                templateHelper.addDecoratedStyleBaseProperties(
                    remoteViews,
                    R.id.expandedRootView,
                    template,
                    metaData
                )
                if (metaData.payload.addOnFeatures.isPersistent) {
                    templateHelper.setDismissCtaCustomization(
                        remoteViews,
                        template.dismissCta
                    )
                }
            } else {
                templateHelper.setHeaderAssetsAndIcon(
                    context,
                    remoteViews,
                    template,
                    metaData
                )
            }
            // default text
            templateHelper.setDefaultTextAndStyle(
                remoteViews,
                template.defaultText,
                getAppName(context),
                template.headerStyle
            )
            templateHelper.addLargeIcon(remoteViews, template, metaData.payload)
            // persistent notification
            if (metaData.payload.addOnFeatures.isPersistent) {
                templateHelper.addActionToDismissCTA(remoteViews, context, metaData)
            }
            templateHelper.addDefaultActionToNotificationClick(
                context,
                remoteViews,
                R.id.expandedRootView,
                template,
                metaData
            )
            metaData.notificationBuilder.setCustomBigContentView(remoteViews)
            return true
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag buildBigTextStyleNotification() :" }
        }
        return false
    }

    private fun updateNotificationClearCallback(
        context: Context,
        template: Template,
        metaData: NotificationMetaData
    ) {
        val finalIntent = getNotificationClearIntent(context, metaData, template)
        setNotificationClearIntent(context, metaData, finalIntent)
    }

    private fun shouldAddBigText(
        template: Template,
        hasCustomCollapsed: Boolean,
        hasCustomExpanded: Boolean
    ): Boolean {
        val collapsedType =
            template.collapsedTemplate?.type ?: return hasCustomCollapsed && !hasCustomExpanded
        if (TEMPLATE_NAME_IMAGE_BANNER.equals(collapsedType)) return false
        return hasCustomCollapsed && !hasCustomExpanded
    }

    private fun shouldBuildTemplate(
        template: Template,
        sdkInstance: SdkInstance,
        progressProperties: ProgressProperties
    ): Boolean {
        if (template is TimerTemplate && progressProperties.timerEndTime == -1L) {
            sdkInstance.logger.log { " shouldBuildTemplate() : template has expired $progressProperties." }
            return false
        }
        return true
    }

    private fun setUpProgressPropertiesIfRequired(
        isTimerTemplate: Boolean,
        template: Template,
        metaData: NotificationMetaData,
        sdkInstance: SdkInstance,
        progressProperties: ProgressProperties
    ): ProgressProperties {
        if (isTimerTemplate) {
            // Unique id for timer alarm trigger
            progressProperties.timerAlarmId = getTimerExpiryAlarmId(metaData)
            // Unique id for progress update alarm trigger
            if (template.collapsedTemplate?.type == TEMPLATE_NAME_TIMER_WITH_PROGRESS) {
                progressProperties.progressAlarmId = getProgressUpdateAlarmId(metaData)
                sdkInstance.logger.log {
                    "$tag buildTemplate() : progressAlarmId: " +
                        "${progressProperties.progressAlarmId}"
                }
            }
            // NOTE: This is required to handle alarm cancel on notification dismiss events
            metaData.payload.payload.putInt(TIMER_ALARM_ID, progressProperties.timerAlarmId)
            metaData.payload.payload.putInt(PROGRESS_ALARM_ID, progressProperties.progressAlarmId)
            // set progress update parameters
            addProgressPropertiesIfRequired(
                progressProperties,
                template,
                metaData,
                sdkInstance
            )
        }
        return progressProperties
    }

    private fun getBigTextStyleRemoteView(context: Context, sdkInstance: SdkInstance): RemoteViews {
        return if (doesSdkSupportDecoratedStyleOnDevice()) {
            RemoteViews(
                context.packageName,
                R.layout.moe_rich_push_stylized_basic_big_text_decorated_style
            )
        } else {
            RemoteViews(
                context.packageName,
                getTemplateLayout(
                    R.layout.moe_rich_push_stylized_basic_big_text,
                    R.layout.moe_rich_push_stylized_basic_big_text_big_layout,
                    sdkInstance
                )
            )
        }
    }

    private fun getMaxMessageLinesToShow(
        isDecoratedCustomViewStyle: Boolean,
        isPersistent: Boolean
    ): Int {
        return if (!isDecoratedCustomViewStyle) {
            // if decorated style is not supported
            13
        } else if (!isPersistent) {
            // if decorated style is supported & template is not persistent
            DECORATED_STYLE_MAX_LINES_WITHOUT_ACTION_IMAGE
        } else {
            // if decorated style is supported & template is persistent
            DECORATED_STYLE_MAX_LINES_WITH_ACTION_WITHOUT_IMAGE
        }
    }
}