/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.richnotification.internal.builder

import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.graphics.Color
import android.os.Build
import android.util.TypedValue
import android.view.View
import android.widget.ImageView
import android.widget.RemoteViews
import androidx.annotation.DrawableRes
import androidx.core.text.HtmlCompat
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.core.internal.utils.downloadImageBitmap
import com.moengage.core.internal.utils.getDeviceDimensions
import com.moengage.core.internal.utils.getPendingIntentActivity
import com.moengage.core.internal.utils.getPendingIntentService
import com.moengage.core.internal.utils.isTablet
import com.moengage.pushbase.ACTIONS
import com.moengage.pushbase.ACTION_REMIND_ME_LATER
import com.moengage.pushbase.internal.ACTION
import com.moengage.pushbase.internal.CacheStrategy
import com.moengage.pushbase.internal.INTENT_ACTION_NOTIFICATION_CLOSE_CLICKED
import com.moengage.pushbase.internal.ImageHelper
import com.moengage.pushbase.internal.MoEPushWorker
import com.moengage.pushbase.internal.TEMPLATE_META
import com.moengage.pushbase.internal.buildDismissActionJson
import com.moengage.pushbase.internal.getIntentForSnooze
import com.moengage.pushbase.internal.getRedirectIntent
import com.moengage.pushbase.internal.model.NotificationMetaData
import com.moengage.pushbase.internal.model.TemplateTrackingMeta
import com.moengage.pushbase.internal.templateTrackingMetaToJsonString
import com.moengage.pushbase.internal.transformToPx
import com.moengage.pushbase.model.NotificationPayload
import com.moengage.pushbase.model.action.Action
import com.moengage.richnotification.R
import com.moengage.richnotification.internal.ASSET_COLOR_DARK_GREY
import com.moengage.richnotification.internal.ASSET_COLOR_LIGHT_GREY
import com.moengage.richnotification.internal.Evaluator
import com.moengage.richnotification.internal.MAX_IMAGE_HEIGHT
import com.moengage.richnotification.internal.MODULE_TAG
import com.moengage.richnotification.internal.WIDGET_TYPE_BUTTON
import com.moengage.richnotification.internal.WIDGET_TYPE_IMAGE
import com.moengage.richnotification.internal.doesSdkSupportDecoratedStyleOnDevice
import com.moengage.richnotification.internal.getTime
import com.moengage.richnotification.internal.models.Card
import com.moengage.richnotification.internal.models.ChronometerStyle
import com.moengage.richnotification.internal.models.DefaultText
import com.moengage.richnotification.internal.models.DismissCta
import com.moengage.richnotification.internal.models.HeaderStyle
import com.moengage.richnotification.internal.models.ImageWidget
import com.moengage.richnotification.internal.models.LayoutStyle
import com.moengage.richnotification.internal.models.Template
import com.moengage.richnotification.internal.models.Widget
import org.json.JSONArray
import org.json.JSONObject
import kotlin.math.min

/**
 * @author Umang Chamaria
 * Date: 11/03/20
 */
private const val BUTTON_HEIGHT = 40

internal class TemplateHelper(private val sdkInstance: SdkInstance) {

    private val tag = "${MODULE_TAG}TemplateHelper"

    fun setAssetsIfRequired(
        remoteViews: RemoteViews,
        template: Template,
        payload: NotificationPayload
    ) {
        when (template.assetColor) {
            ASSET_COLOR_DARK_GREY -> setAssets(
                remoteViews,
                payload.addOnFeatures.isPersistent,
                template.defaultText,
                R.drawable.moe_rich_push_dark_cross,
                R.drawable.moe_rich_push_dark_separator
            )
            ASSET_COLOR_LIGHT_GREY -> setAssets(
                remoteViews,
                payload.addOnFeatures.isPersistent,
                template.defaultText,
                R.drawable.moe_rich_push_light_cross,
                R.drawable.moe_rich_push_light_separator
            )
            else -> {
                sdkInstance.logger.log(LogLevel.ERROR) { "$tag setAssetsIfRequired() : Not a valid asset color, using default." }
                setAssets(
                    remoteViews,
                    payload.addOnFeatures.isPersistent,
                    template.defaultText,
                    R.drawable.moe_rich_push_light_cross,
                    R.drawable.moe_rich_push_light_separator
                )
            }
        }
    }

    private fun setAssets(
        remoteViews: RemoteViews,
        isPersistent: Boolean,
        defaultText: DefaultText,
        @DrawableRes crossButton: Int,
        @DrawableRes separator: Int
    ) {
        if (isPersistent) {
            remoteViews.setImageViewResource(R.id.closeButton, crossButton)
            remoteViews.setViewVisibility(R.id.closeButton, View.VISIBLE)
        }
        if (defaultText.summary.isNotBlank()) {
            remoteViews.setImageViewResource(R.id.separatorSummary, separator)
            remoteViews.setViewVisibility(R.id.separatorSummary, View.VISIBLE)
        }
        remoteViews.setImageViewResource(R.id.separatorTime, separator)
    }

    @Throws(IllegalStateException::class)
    fun setDefaultTextAndStyle(
        remoteViews: RemoteViews,
        defaultText: DefaultText,
        appName: String,
        headerStyle: HeaderStyle
    ) {
        remoteViews.setTextViewText(
            R.id.title,
            HtmlCompat.fromHtml(defaultText.title, HtmlCompat.FROM_HTML_MODE_COMPACT).trim()
        )
        remoteViews.setTextViewText(
            R.id.message,
            HtmlCompat.fromHtml(defaultText.message, HtmlCompat.FROM_HTML_MODE_COMPACT).trim()
        )
        if (!doesSdkSupportDecoratedStyleOnDevice()) {
            setHeaderTextAndStyle(
                remoteViews,
                defaultText,
                appName,
                headerStyle
            )
        }
    }

    fun addActionToDismissCTA(
        remoteViews: RemoteViews,
        context: Context,
        metaData: NotificationMetaData
    ) {
        val intent =
            Intent(context, MoEPushWorker::class.java)
        intent.putExtras(metaData.payload.payload)
            .putExtra(ACTION, buildDismissActionJson(metaData.notificationId).toString())
            .action = INTENT_ACTION_NOTIFICATION_CLOSE_CLICKED
        val pendingIntent = getPendingIntentService(context, metaData.notificationId, intent)
        remoteViews.setOnClickPendingIntent(R.id.closeButton, pendingIntent)
    }

    fun addLargeIcon(remoteViews: RemoteViews, template: Template, payload: NotificationPayload) {
        var largeIcon: Bitmap? = null
        if (!template.shouldShowLargeIcon) return
        if (payload.addOnFeatures.largeIconUrl.isNotBlank()) {
            largeIcon =
                ImageHelper(sdkInstance).getBitmapFromUrl(payload.addOnFeatures.largeIconUrl, CacheStrategy.MEMORY)
        }
        if (largeIcon != null) {
            remoteViews.setImageViewBitmap(R.id.largeIcon, largeIcon)
        } else {
            if (sdkInstance.initConfig.push.meta.largeIcon != -1) {
                remoteViews.setImageViewResource(
                    R.id.largeIcon,
                    sdkInstance.initConfig.push.meta.largeIcon
                )
            }
        }
        if (doesSdkSupportDecoratedStyleOnDevice()) {
            setViewCornerToRounded(remoteViews, R.id.largeIcon)
        }
        remoteViews.setViewVisibility(R.id.largeIcon, View.VISIBLE)
    }

    fun setBackgroundColor(layout: LayoutStyle, remoteViews: RemoteViews, viewId: Int) {
        if (layout.backgroundColor.isBlank()) return
        remoteViews.setInt(viewId, "setBackgroundColor", Color.parseColor(layout.backgroundColor))
    }

    fun scaleBitmap(context: Context, bitmap: Bitmap, maxHeight: Int): Bitmap {
        try {
            val imageHeight = bitmap.height
            val imageWidth = bitmap.width
            val displayMetrics = context.resources.displayMetrics
            sdkInstance.logger.log { "$tag scaleBitmap() : Max height: $maxHeight" }
            sdkInstance.logger.log { "$tag scaleBitmap() : Device dimensions: width: ${displayMetrics.widthPixels} height: ${displayMetrics.heightPixels}" }
            sdkInstance.logger.log { "$tag scaleBitmap() : Actual Dimension - width: $imageWidth   height: $imageHeight" }
            return if (imageHeight >= imageWidth) {
                var width = imageWidth * maxHeight / imageHeight
                if (width > displayMetrics.widthPixels) {
                    width = displayMetrics.widthPixels
                }
                sdkInstance.logger.log { "$tag scaleBitmap() : Scaled dimensions: width: $width height: $maxHeight" }
                Bitmap.createScaledBitmap(bitmap, width, maxHeight, true)
            } else {
                val height = imageHeight * displayMetrics.widthPixels / imageWidth
                sdkInstance.logger.log { "$tag scaleBitmap() : Scaled dimensions: width: ${displayMetrics.widthPixels} height: $height" }
                Bitmap.createScaledBitmap(bitmap, displayMetrics.widthPixels, height, true)
            }
        } catch (t: Throwable) {
            sdkInstance.logger.log(LogLevel.ERROR, t) { "$tag scaleBitmap() : " }
        }
        return bitmap
    }

    fun addActionToCard(
        context: Context,
        metaData: NotificationMetaData,
        templateName: String,
        remoteViews: RemoteViews,
        card: Card,
        viewId: Int
    ) {
        if (card.actions.isEmpty()) return
        val intent = getRedirectIntent(context, metaData.payload.payload, metaData.notificationId)
        val meta = TemplateTrackingMeta(templateName, card.id, -1)
        intent.putExtra(TEMPLATE_META, templateTrackingMetaToJsonString(meta))
            .putExtra(ACTION, actionListToActionJson(card.actions).toString())
        val pendingIntent =
            getPendingIntentActivity(context, metaData.notificationId + (1000 + card.id), intent)
        remoteViews.setOnClickPendingIntent(viewId, pendingIntent)
    }

    fun addActionToWidget(
        context: Context,
        metaData: NotificationMetaData,
        templateName: String,
        remoteViews: RemoteViews,
        card: Card,
        widget: Widget,
        viewId: Int
    ) {
        if (widget.actions.isEmpty()) return
        val intent = getRedirectIntent(context, metaData.payload.payload, metaData.notificationId)
        val templateHelper = TemplateHelper(sdkInstance)
        intent.putExtra(
            TEMPLATE_META,
            templateTrackingMetaToJsonString(
                TemplateTrackingMeta(
                    templateName,
                    card.id,
                    widget.id
                )
            )
        )
            .putExtra(ACTION, templateHelper.actionListToActionJson(widget.actions).toString())

        val pendingIntent =
            getPendingIntentActivity(context, metaData.notificationId + (100 + widget.id), intent)
        remoteViews.setOnClickPendingIntent(viewId, pendingIntent)
    }

    fun addLayoutStyle(layout: LayoutStyle?, remoteViews: RemoteViews, viewId: Int) {
        if (layout == null) return
        setBackgroundColor(layout, remoteViews, viewId)
    }

    fun addPersistenceAsset(assetColor: String, remoteViews: RemoteViews, viewId: Int) {
        val assetDrawable = if (ASSET_COLOR_DARK_GREY == assetColor) {
            R.drawable.moe_rich_push_dark_cross
        } else {
            R.drawable.moe_rich_push_light_cross
        }
        remoteViews.setImageViewResource(viewId, assetDrawable)
        remoteViews.setViewVisibility(viewId, View.VISIBLE)
    }

    fun setSmallIconColor(context: Context, remoteViews: RemoteViews) {
        if (sdkInstance.initConfig.push.meta.notificationColor <= 0) return
        remoteViews.setInt(
            R.id.smallIcon,
            "setColorFilter",
            context.resources.getColor(sdkInstance.initConfig.push.meta.notificationColor)
        )
    }

    fun actionListToActionJson(actions: Array<Action>): JSONObject {
        val actionJson = JSONObject()
        val actionsArray = JSONArray()
        if (actions.isEmpty()) {
            actionJson.put(ACTIONS, actionsArray)
            return actionJson
        }
        for (action in actions) {
            actionsArray.put(action.payload)
        }
        actionJson.put(ACTIONS, actionsArray)
        return actionJson
    }

    fun setHeaderStyle(remoteViews: RemoteViews, headerStyle: HeaderStyle) {
        if (!headerStyle.appNameColor.isNullOrBlank()) {
            val color = Color.parseColor(headerStyle.appNameColor)
            remoteViews.setTextColor(R.id.appName, color)
            remoteViews.setTextColor(R.id.time, color)
        }
    }

    fun setContentText(remoteViews: RemoteViews, defaultText: DefaultText) {
        remoteViews.setTextViewText(
            R.id.title,
            HtmlCompat.fromHtml(defaultText.title, HtmlCompat.FROM_HTML_MODE_COMPACT).trim()
        )
        // NOTE: message will be blank in progress template
        if (defaultText.message.isNotBlank()) {
            remoteViews.setTextViewText(
                R.id.message,
                HtmlCompat.fromHtml(defaultText.message, HtmlCompat.FROM_HTML_MODE_COMPACT).trim()
            )
        }
    }

    internal fun addActionButton(
        context: Context,
        metaData: NotificationMetaData,
        template: Template,
        remoteViews: RemoteViews,
        actionButtons: List<Widget>,
        isPersistent: Boolean
    ) {
        if (actionButtons.isNotEmpty()) {
            val maxButtonWidth = getDeviceDimensions(context).width / actionButtons.size
            for (i in 0 until min(actionButtons.size, 2)) {
                val widget = actionButtons[i]
                check(WIDGET_TYPE_BUTTON == widget.type) { "Only button widget expected." }
                remoteViews.setViewVisibility(actionButtonIdArray[i], View.VISIBLE)
                if (!doesSdkSupportDecoratedStyleOnDevice()) {
                    remoteViews.setInt(actionButtonIdArray[i], "setMaxWidth", maxButtonWidth)
                }
                remoteViews.setTextViewText(
                    actionButtonIdArray[i],
                    HtmlCompat.fromHtml(widget.content, HtmlCompat.FROM_HTML_MODE_COMPACT)
                )
                if (widget.style != null && widget.style.backgroundColor.isNotBlank()) {
                    remoteViews.setInt(
                        actionButtonIdArray[i],
                        "setBackgroundColor",
                        Color.parseColor(widget.style.backgroundColor)
                    )
                }
                val meta = TemplateTrackingMeta(template.templateName, -1, widget.id)
                var redirectIntent =
                    getRedirectIntent(context, metaData.payload.payload, metaData.notificationId)
                if (containsSnoozeAction(widget.actions)) {
                    redirectIntent =
                        getIntentForSnooze(
                            context,
                            metaData.payload.payload,
                            metaData.notificationId
                        )
                }
                redirectIntent.putExtra(TEMPLATE_META, templateTrackingMetaToJsonString(meta))
                if (widget.actions.isNotEmpty()) {
                    val templateHelper = TemplateHelper(sdkInstance)
                    redirectIntent.putExtra(
                        ACTION,
                        templateHelper.actionListToActionJson(widget.actions).toString()
                    )
                }
                val pendingIntent = getPendingIntentActivity(
                    context,
                    metaData.notificationId + (1000 + widget.id),
                    redirectIntent
                )
                remoteViews.setOnClickPendingIntent(actionButtonIdArray[i], pendingIntent)
            }
        }
        // add dismiss button
        if (isPersistent) {
            val shouldCustomiseDismissView = doesSdkSupportDecoratedStyleOnDevice() ||
                Evaluator(sdkInstance.logger).isTimerTemplate(
                    template.collapsedTemplate?.type,
                    template.expandedTemplate?.type
                )
            setDismissCtaCustomization(
                remoteViews,
                template.dismissCta,
                shouldCustomiseDismissView
            )
            addActionToDismissCTA(remoteViews, context, metaData)
        }
    }

    private val actionButtonIdArray = intArrayOf(
        R.id.actionButton1,
        R.id.actionButton2
    )

    private fun containsSnoozeAction(actions: Array<Action>?): Boolean {
        if (actions == null) return false
        for (action in actions) {
            if (action.actionType == ACTION_REMIND_ME_LATER) {
                return true
            }
        }
        return false
    }

    internal fun addImageToExpandedTemplate(
        context: Context,
        metaData: NotificationMetaData,
        template: Template,
        remoteViews: RemoteViews
    ): Boolean {
        if (template.expandedTemplate == null) return false

        val card = template.expandedTemplate.cards[0]
        if (card.widgets.isEmpty()) return false
        val widget = card.widgets[0]
        if (WIDGET_TYPE_IMAGE != widget.type) return false

        return addImageWidgetToTemplate(
            context,
            metaData,
            template,
            remoteViews,
            widget as ImageWidget,
            card
        )
    }

    private fun addDefaultAction(
        context: Context,
        template: Template,
        metaData: NotificationMetaData,
        card: Card,
        remoteViews: RemoteViews,
        widgetId: Int
    ) {
        val meta = TemplateTrackingMeta(template.templateName, card.id, -1)
        val redirectIntent =
            getRedirectIntent(context, metaData.payload.payload, metaData.notificationId)
        redirectIntent.putExtra(TEMPLATE_META, templateTrackingMetaToJsonString(meta))
        // click intent
        val pendingIntent =
            getPendingIntentActivity(context, metaData.notificationId, redirectIntent)
        remoteViews.setOnClickPendingIntent(widgetId, pendingIntent)
    }

    internal fun setChronometer(
        remoteViews: RemoteViews,
        format: String,
        timerExpiry: Long
    ): Boolean {
        if (timerExpiry == -1L) return false
        remoteViews.setChronometer(R.id.moEChronometer, timerExpiry, format, true)
        remoteViews.setViewVisibility(R.id.chronometerLayout, View.VISIBLE)
        remoteViews.setViewVisibility(R.id.moEChronometer, View.VISIBLE)
        return true
    }

    internal fun getChronometerStyle(widget: Widget): ChronometerStyle? {
        return if (widget.style !is ChronometerStyle) {
            return null
        } else {
            widget.style
        }
    }

    internal fun addImageWidgetToTemplate(
        context: Context,
        metaData: NotificationMetaData,
        template: Template,
        remoteViews: RemoteViews,
        widget: ImageWidget,
        card: Card,
        preloadedBitmap: Bitmap? = null,
        maxAllowedImageHeight: Int = MAX_IMAGE_HEIGHT
    ): Boolean {
        if (template.expandedTemplate == null) return false
        var bitmap: Bitmap = preloadedBitmap ?: downloadImageBitmap(widget.content) ?: return false
        val widgetId: Int
        if (doesSdkSupportDecoratedStyleOnDevice()) {
            widgetId = if (widget.scaleType == ImageView.ScaleType.CENTER_CROP) {
                remoteViews.setViewVisibility(R.id.centerInsideImage, View.GONE)
                setViewCornerToRounded(remoteViews, R.id.centerCropImage)
                R.id.centerCropImage
            } else {
                remoteViews.setViewVisibility(R.id.centerCropImage, View.GONE)
                R.id.centerInsideImage
            }
        } else {
            val maxImageHeight = if (template.expandedTemplate.actionButtonList.isNotEmpty()) {
                transformToPx(context, maxAllowedImageHeight - BUTTON_HEIGHT)
            } else {
                transformToPx(context, maxAllowedImageHeight)
            }
            val isTablet = isTablet(context)
            if (!isTablet) {
                // Don't scale the bitmap for Tablet
                bitmap = scaleBitmap(
                    context,
                    bitmap,
                    maxImageHeight
                )
            }
            widgetId = if (isTablet) {
                // Use `scaleType` `centerCrop` for Tablets
                remoteViews.setViewVisibility(R.id.horizontalFitCenterImage, View.GONE)
                remoteViews.setViewVisibility(R.id.verticalImage, View.GONE)
                R.id.horizontalCenterCropImage
            } else if (bitmap.height >= bitmap.width) {
                remoteViews.setViewVisibility(R.id.horizontalCenterCropImage, View.GONE)
                remoteViews.setViewVisibility(R.id.horizontalFitCenterImage, View.GONE)
                R.id.verticalImage
            } else {
                if (bitmap.height >= maxImageHeight) {
                    remoteViews.setViewVisibility(R.id.horizontalFitCenterImage, View.GONE)
                    remoteViews.setViewVisibility(R.id.verticalImage, View.GONE)
                    R.id.horizontalCenterCropImage
                } else {
                    remoteViews.setViewVisibility(R.id.horizontalCenterCropImage, View.GONE)
                    remoteViews.setViewVisibility(R.id.verticalImage, View.GONE)
                    R.id.horizontalFitCenterImage
                }
            }
        }
        remoteViews.setImageViewBitmap(widgetId, bitmap)
        remoteViews.setViewVisibility(widgetId, View.VISIBLE)
        addActionToImageWidget(context, metaData, template, remoteViews, widget, card, widgetId)
        return true
    }

    internal fun removeImageWidgetFromExpandedTemplate(remoteViews: RemoteViews) {
        if (doesSdkSupportDecoratedStyleOnDevice()) {
            remoteViews.setViewVisibility(R.id.centerInsideImage, View.GONE)
            remoteViews.setViewVisibility(R.id.centerCropImage, View.GONE)
        } else {
            remoteViews.setViewVisibility(R.id.horizontalFitCenterImage, View.GONE)
            remoteViews.setViewVisibility(R.id.verticalImage, View.GONE)
            remoteViews.setViewVisibility(R.id.horizontalCenterCropImage, View.GONE)
        }
    }

    private fun setHeaderTextAndStyle(
        remoteViews: RemoteViews,
        defaultText: DefaultText,
        appName: String,
        headerStyle: HeaderStyle
    ) {
        if (defaultText.summary.isNotBlank()) {
            remoteViews.setViewVisibility(
                R.id.summaryText,
                View.VISIBLE
            )
            remoteViews.setTextViewText(
                R.id.summaryText,
                HtmlCompat.fromHtml(defaultText.summary, HtmlCompat.FROM_HTML_MODE_COMPACT)
            )
        }
        remoteViews.setTextViewText(R.id.time, getTime())
        if (appName.isBlank()) {
            throw IllegalStateException("App name cannot be empty")
        }
        remoteViews.setTextViewText(
            R.id.appName,
            appName
        )
        setHeaderStyle(remoteViews, headerStyle)
    }

    // Helper function to make the layout corner as rounder corner (works only for >= S)
    internal fun setViewCornerToRounded(
        remoteViews: RemoteViews,
        layoutId: Int,
        radius: Float = 4f,
        unit: Int = TypedValue.COMPLEX_UNIT_DIP
    ) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
            remoteViews.setViewOutlinePreferredRadius(layoutId, radius, unit)
        }
    }

    internal fun setDismissCtaCustomization(
        remoteViews: RemoteViews,
        dismissCtaText: DismissCta,
        shouldCustomiseDismissView: Boolean = doesSdkSupportDecoratedStyleOnDevice()
    ) {
        if (shouldCustomiseDismissView) {
            remoteViews.setTextViewText(
                R.id.closeButton,
                HtmlCompat.fromHtml(dismissCtaText.text, HtmlCompat.FROM_HTML_MODE_COMPACT)
            )
        }
        remoteViews.setViewVisibility(R.id.closeButton, View.VISIBLE)
    }

    internal fun setHeaderAssetsAndIcon(
        context: Context,
        remoteViews: RemoteViews,
        template: Template,
        metaData: NotificationMetaData
    ) {
        // set assets
        setAssetsIfRequired(remoteViews, template, metaData.payload)
        // small icon
        if (sdkInstance.initConfig.push.meta.smallIcon != -1) {
            remoteViews.setImageViewResource(
                R.id.smallIcon,
                sdkInstance.initConfig.push.meta.smallIcon
            )
            setSmallIconColor(context, remoteViews)
        }
    }

    internal fun addDecoratedStyleBaseProperties(
        remoteViews: RemoteViews,
        rootViewId: Int,
        template: Template,
        metaData: NotificationMetaData
    ) {
        setViewCornerToRounded(remoteViews, rootViewId)
        metaData.notificationBuilder.setSubText(
            HtmlCompat.fromHtml(template.defaultText.summary, HtmlCompat.FROM_HTML_MODE_COMPACT)
                .trim()
        )
    }

    internal fun addDefaultActionToNotificationClick(
        context: Context,
        remoteViews: RemoteViews,
        rootViewId: Int,
        template: Template,
        metaData: NotificationMetaData
    ) {
        val meta = TemplateTrackingMeta(template.templateName, -1, -1)
        val redirectIntent =
            getRedirectIntent(context, metaData.payload.payload, metaData.notificationId)
        redirectIntent.putExtra(TEMPLATE_META, templateTrackingMetaToJsonString(meta))
        // click intent
        val intent = getPendingIntentActivity(context, metaData.notificationId, redirectIntent)
        remoteViews.setOnClickPendingIntent(rootViewId, intent)
        metaData.notificationBuilder.setContentIntent(intent)
    }

    internal fun addActionToImageWidget(
        context: Context,
        metaData: NotificationMetaData,
        template: Template,
        remoteViews: RemoteViews,
        widget: ImageWidget,
        card: Card,
        widgetId: Int,
        cardId: Int = R.id.card
    ) {
        if (widget.actions.isEmpty() && card.actions.isEmpty()) {
            addDefaultAction(context, template, metaData, card, remoteViews, widgetId)
        } else {
            // widget action
            addActionToWidget(
                context,
                metaData,
                template.templateName,
                remoteViews,
                card,
                widget,
                widgetId
            )
            // card action
            addActionToCard(
                context,
                metaData,
                template.templateName,
                remoteViews,
                card,
                cardId
            )
        }
    }
}