package com.moengage.richnotification.internal.builder

import android.annotation.TargetApi
import android.content.Context
import android.graphics.Color
import android.os.Build
import android.os.SystemClock
import android.view.View
import android.widget.RemoteViews
import androidx.annotation.RequiresApi
import com.moengage.core.LogLevel
import com.moengage.core.internal.model.SdkInstance
import com.moengage.pushbase.internal.model.NotificationMetaData
import com.moengage.richnotification.R
import com.moengage.richnotification.internal.BIG_TEXT_MAX_LINES_WITHOUT_ACTION_BUTTONS
import com.moengage.richnotification.internal.BIG_TEXT_MAX_LINES_WITH_ACTION_BUTTONS
import com.moengage.richnotification.internal.DECORATED_STYLE_MAX_LINES_WITHOUT_ACTION_IMAGE
import com.moengage.richnotification.internal.DECORATED_STYLE_MAX_LINES_WITH_ACTION_IMAGE
import com.moengage.richnotification.internal.DECORATED_STYLE_MAX_LINES_WITH_ACTION_WITHOUT_IMAGE
import com.moengage.richnotification.internal.Evaluator
import com.moengage.richnotification.internal.MODULE_TAG
import com.moengage.richnotification.internal.PROGRESS_BAR_TEMPLATE_MAX_PROGRESS_VALUE
import com.moengage.richnotification.internal.TIMER_FORMAT_HH_MM
import com.moengage.richnotification.internal.TIMER_FORMAT_H_MM_SS
import com.moengage.richnotification.internal.WIDGET_TYPE_IMAGE
import com.moengage.richnotification.internal.doesSdkSupportDecoratedStyleOnDevice
import com.moengage.richnotification.internal.getDecoratedStyleTemplateLayout
import com.moengage.richnotification.internal.getTimerEndTime
import com.moengage.richnotification.internal.hasScheduleExactPermission
import com.moengage.richnotification.internal.models.ChronometerWidget
import com.moengage.richnotification.internal.models.ImageWidget
import com.moengage.richnotification.internal.models.ProgressProperties
import com.moengage.richnotification.internal.models.ProgressbarWidget
import com.moengage.richnotification.internal.models.TimerTemplate

/**
 * @author Arshiya Khanum
 */

private val TIMER_FORMAT_TO_STRING_MAPPING = mapOf(
    TIMER_FORMAT_HH_MM to "%s:%s",
    TIMER_FORMAT_H_MM_SS to "%s:%s:%s"
)

internal class TimerTemplateBuilder(
    private val context: Context,
    private val template: TimerTemplate,
    private val metaData: NotificationMetaData,
    private val sdkInstance: SdkInstance,
    private val progressProperties: ProgressProperties
) {
    private val tag = "${MODULE_TAG}TimerTemplateBuilder"
    private val templateHelper = TemplateHelper(sdkInstance)

    @TargetApi(Build.VERSION_CODES.N)
    fun buildCollapsedTimerTemplate(): Boolean {
        if (template.collapsedTemplate == null) return false
        if (!Evaluator(sdkInstance.logger).hasMinimumText(template.defaultText)) {
            sdkInstance.logger.log(LogLevel.WARN) { "$tag buildCollapsedTimerTemplate() : Does not have minimum text." }
            return false
        }
        sdkInstance.logger.log { "$tag buildCollapsedTimerTemplate() : Template: ${template.collapsedTemplate}" }

        if (template.collapsedTemplate.cards.isEmpty()) return false
        val remoteViews = getTimerCollapsedRemoteViews()
        templateHelper.setContentText(remoteViews, template.defaultText)
        // check for timer
        if (template.collapsedTemplate.cards.isNotEmpty()) {
            val card = template.collapsedTemplate.cards[0]
            for (widget in card.widgets) {
                if (widget.id == 1 && widget is ChronometerWidget) {
                    // check for timer
                    checkAndAddChronometer(
                        remoteViews,
                        widget
                    )
                }
            }
        }
        // add default action
        templateHelper.addDefaultActionToNotificationClick(
            context,
            remoteViews,
            R.id.collapsedRootView,
            template,
            metaData
        )
        metaData.notificationBuilder.setCustomContentView(remoteViews)
        return true
    }

    private fun getTimerCollapsedRemoteViews(): RemoteViews {
        return RemoteViews(
            context.packageName,
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                getDecoratedStyleTemplateLayout(
                    R.layout.moe_rich_push_timer_collapsed_small_layout_decorated_style,
                    R.layout.moe_rich_push_timer_collapsed_layout_decorated_style,
                    sdkInstance
                )
            } else {
                R.layout.moe_rich_push_timer_collapsed_layout
            }
        )
    }

    @TargetApi(Build.VERSION_CODES.N)
    fun buildExpandedTimerTemplate(): Boolean {
        if (template.expandedTemplate == null) return false
        if (!Evaluator(sdkInstance.logger).hasMinimumText(template.defaultText)) {
            sdkInstance.logger.log(LogLevel.WARN) { "$tag buildExpandedTimerTemplate() : Does not have minimum text." }
            return false
        }
        sdkInstance.logger.log { "$tag buildExpandedTimerTemplate() : Template: ${template.expandedTemplate}" }
        if (template.expandedTemplate.cards.isEmpty()) return false
        val hasActionButtons =
            template.expandedTemplate.actionButtonList.isNotEmpty() || metaData.payload.addOnFeatures.isPersistent
        val remoteViews = getTimerExpandedRemoteViews(hasActionButtons)
        if (template.expandedTemplate.cards.isEmpty() &&
            template.expandedTemplate.actionButtonList.isEmpty()
        ) {
            return false
        }

        // set title and message
        templateHelper.setContentText(remoteViews, template.defaultText)
        // add action buttons
        if (hasActionButtons) {
            templateHelper.addActionButton(
                context,
                metaData,
                template,
                remoteViews,
                template.expandedTemplate.actionButtonList,
                metaData.payload.addOnFeatures.isPersistent
            )
        }
        var hasImage = false
        if (template.expandedTemplate.cards.isNotEmpty()) {
            val card = template.expandedTemplate.cards[0]
            for (widget in card.widgets) {
                if (widget.id == 0 && widget.type == WIDGET_TYPE_IMAGE) {
                    // if image is available
                    hasImage = templateHelper.addImageWidgetToTemplate(
                        context,
                        metaData,
                        template,
                        remoteViews,
                        widget as ImageWidget,
                        card
                    )
                } else if (widget.id == 1 && widget is ChronometerWidget) {
                    // check for timer
                    checkAndAddChronometer(
                        remoteViews,
                        widget
                    )
                }
            }
        }
        // set max lines for content
        setMessageMaxLines(
            remoteViews,
            hasActionButtons,
            hasImage
        )
        // add default click action
        templateHelper.addDefaultActionToNotificationClick(
            context,
            remoteViews,
            R.id.expandedRootView,
            template,
            metaData
        )
        metaData.notificationBuilder.setCustomBigContentView(remoteViews)
        return true
    }

    private fun getTimerExpandedRemoteViews(hasButtons: Boolean): RemoteViews {
        return RemoteViews(
            context.packageName,
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                if (hasButtons) {
                    R.layout.moe_rich_push_timer_expanded_decorated_style_with_action_buttons
                } else {
                    R.layout.moe_rich_push_timer_expanded_decorated_style_without_action_buttons
                }
            } else {
                if (hasButtons) {
                    R.layout.moe_rich_push_timer_expanded_with_action_buttons
                } else {
                    R.layout.moe_rich_push_timer_expanded_without_action_buttons
                }
            }
        )
    }

    private fun setMessageMaxLines(
        remoteViews: RemoteViews,
        hasActionButtons: Boolean,
        hasImage: Boolean
    ) {
        if (doesSdkSupportDecoratedStyleOnDevice()) {
            remoteViews.setInt(
                R.id.message,
                "setMaxLines",
                if (hasImage) {
                    DECORATED_STYLE_MAX_LINES_WITH_ACTION_IMAGE
                } else if (hasActionButtons) {
                    DECORATED_STYLE_MAX_LINES_WITH_ACTION_WITHOUT_IMAGE
                } else {
                    DECORATED_STYLE_MAX_LINES_WITHOUT_ACTION_IMAGE
                }
            )
        } else {
            if (hasImage) {
                remoteViews.setBoolean(R.id.message, "setSingleLine", true)
                remoteViews.setInt(R.id.message, "setMaxLines", 1)
                return
            }
            remoteViews.setBoolean(R.id.message, "setSingleLine", false)
            remoteViews.setInt(
                R.id.message,
                "setMaxLines",
                if (hasActionButtons) {
                    // BigText with action button
                    BIG_TEXT_MAX_LINES_WITH_ACTION_BUTTONS
                } else {
                    BIG_TEXT_MAX_LINES_WITHOUT_ACTION_BUTTONS
                }
            )
        }
    }

    @RequiresApi(Build.VERSION_CODES.N)
    internal fun setChronometerCountdown(remoteViews: RemoteViews) {
        remoteViews.setChronometerCountDown(R.id.moEChronometer, true)
    }

    @RequiresApi(Build.VERSION_CODES.N)
    private fun checkAndAddChronometer(
        remoteViews: RemoteViews,
        chronometerWidget: ChronometerWidget
    ) {
        setChronometerCountdown(remoteViews)
        setChronometerStyle(chronometerWidget, remoteViews)
        TIMER_FORMAT_TO_STRING_MAPPING[chronometerWidget.properties.format]
            ?.let {
                sdkInstance.logger.log { "$tag checkAndAddChronometer(): format: $it" }
                templateHelper.setChronometer(
                    remoteViews,
                    it,
                    SystemClock.elapsedRealtime() + getTimerEndTime(
                        progressProperties.timerProperties.duration,
                        progressProperties.timerProperties.expiry
                    )
                )
            }
    }

    private fun setChronometerStyle(
        chronometerWidget: ChronometerWidget,
        remoteViews: RemoteViews
    ) {
        val chronometerStyle = templateHelper.getChronometerStyle(chronometerWidget) ?: return
        if (!chronometerStyle.textColor.isNullOrBlank()) {
            remoteViews.setTextColor(
                R.id.moEChronometer,
                Color.parseColor(chronometerStyle.textColor)
            )
        }
    }

    @TargetApi(Build.VERSION_CODES.N)
    internal fun buildCollapsedProgressTemplate(): Boolean {
        if (template.collapsedTemplate == null) return false
        if (template.defaultText.title.isBlank()) {
            sdkInstance.logger.log(LogLevel.WARN) { "$tag buildCollapsedProgressTemplate() : Does not have minimum text." }
            return false
        }
        sdkInstance.logger.log { "$tag buildCollapsedProgressTemplate() : Template: ${template.collapsedTemplate}" }

        if (template.collapsedTemplate.cards.isEmpty()) return false
        val remoteViews = getProgressbarCollapsedRemoteViews()
        templateHelper.setContentText(remoteViews, template.defaultText)
        // check for timer and progress widget
        if (template.collapsedTemplate.cards.isNotEmpty()) {
            val card = template.collapsedTemplate.cards[0]
            for (widget in card.widgets) {
                if (widget.id == 1 && widget is ChronometerWidget) {
                    // check for timer
                    checkAndAddChronometer(
                        remoteViews,
                        widget
                    )
                } else if (widget.id == 2 && widget is ProgressbarWidget) {
                    checkAndAddProgressbar(remoteViews)
                }
            }
        }
        // add default click action
        templateHelper.addDefaultActionToNotificationClick(
            context,
            remoteViews,
            R.id.collapsedRootView,
            template,
            metaData
        )
        metaData.notificationBuilder.setCustomContentView(remoteViews)
        return true
    }

    private fun getProgressbarCollapsedRemoteViews(): RemoteViews {
        return RemoteViews(
            context.packageName,
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                if (hasScheduleExactPermission(context)) {
                    getDecoratedStyleTemplateLayout(
                        R.layout.moe_rich_push_progressbar_collapsed_small_layout_decroated_style,
                        R.layout.moe_rich_push_progressbar_collapsed_layout_decroated_style,
                        sdkInstance
                    )
                } else {
                    getDecoratedStyleTemplateLayout(
                        R.layout.moe_rich_push_timer_with_title_collapsed_small_layout_decroated_style,
                        R.layout.moe_rich_push_timer_with_title_collapsed_layout_decroated_style,
                        sdkInstance
                    )
                }
            } else {
                R.layout.moe_rich_push_progressbar_collapsed_layout
            }
        )
    }

    @TargetApi(Build.VERSION_CODES.N)
    fun buildExpandedProgressTemplate(): Boolean {
        if (template.expandedTemplate == null) return false
        if (template.defaultText.title.isBlank()) {
            sdkInstance.logger.log(LogLevel.WARN) { "$tag buildExpandedProgressTemplate() : Does not have minimum text." }
            return false
        }
        sdkInstance.logger.log { "$tag buildExpandedProgressTemplate() : Template: ${template.expandedTemplate}" }
        if (template.expandedTemplate.cards.isEmpty()) return false

        val hasActionButtons =
            template.expandedTemplate.actionButtonList.isNotEmpty() || metaData.payload.addOnFeatures.isPersistent
        val remoteViews = getProgressbarExpandedRemoteViews(hasActionButtons)

        if (template.expandedTemplate.cards.isEmpty() &&
            template.expandedTemplate.actionButtonList.isEmpty()
        ) {
            return false
        }

        // set title
        templateHelper.setContentText(remoteViews, template.defaultText)
        // add action buttons
        if (hasActionButtons) {
            templateHelper.addActionButton(
                context,
                metaData,
                template,
                remoteViews,
                template.expandedTemplate.actionButtonList,
                metaData.payload.addOnFeatures.isPersistent
            )
        }

        var hasImage = false
        if (template.expandedTemplate.cards.isNotEmpty()) {
            val card = template.expandedTemplate.cards[0]
            for (widget in card.widgets) {
                if (widget.id == 0 && widget.type == WIDGET_TYPE_IMAGE) {
                    // if image is available
                    hasImage = templateHelper.addImageWidgetToTemplate(
                        context,
                        metaData,
                        template,
                        remoteViews,
                        widget as ImageWidget,
                        card
                    )
                } else if (widget.id == 1 && widget is ChronometerWidget) {
                    // check for timer
                    checkAndAddChronometer(
                        remoteViews,
                        widget
                    )
                } else if (widget.id == 2 && widget is ProgressbarWidget) {
                    checkAndAddProgressbar(remoteViews)
                }
            }
        }

        // set max lines for content
        setMessageMaxLines(
            remoteViews,
            hasActionButtons,
            hasImage
        )
        // add default click action
        templateHelper.addDefaultActionToNotificationClick(
            context,
            remoteViews,
            R.id.expandedRootView,
            template,
            metaData
        )
        metaData.notificationBuilder.setCustomBigContentView(remoteViews)
        return true
    }

    private fun getProgressbarExpandedRemoteViews(hasButtons: Boolean): RemoteViews {
        return RemoteViews(
            context.packageName,
            if (doesSdkSupportDecoratedStyleOnDevice()) {
                getDecoratedStyleExpandedProgressbarLayoutId(
                    hasButtons,
                    hasScheduleExactPermission(context)
                )
            } else {
                if (hasButtons) {
                    R.layout.moe_rich_push_progressbar_expanded_with_action_buttons
                } else {
                    R.layout.moe_rich_push_progressbar_expanded_without_action_buttons
                }
            }
        )
    }

    private fun checkAndAddProgressbar(remoteViews: RemoteViews) {
        if (!hasScheduleExactPermission(context)) {
            sdkInstance.logger.log(LogLevel.DEBUG) { "$tag buildExpandedProgressTemplate() : Does not have permission to schedule exact alarm." }
            return
        }
        if (progressProperties.currentProgress <= -1) {
            remoteViews.setViewVisibility(R.id.moEProgressbar, View.GONE)
            return
        }
        remoteViews.setViewVisibility(R.id.chronometerLayout, View.VISIBLE)
        remoteViews.setViewVisibility(R.id.moEProgressbar, View.VISIBLE)
        remoteViews.setProgressBar(
            R.id.moEProgressbar,
            PROGRESS_BAR_TEMPLATE_MAX_PROGRESS_VALUE,
            progressProperties.currentProgress,
            false
        )
    }

    private fun getDecoratedStyleExpandedProgressbarLayoutId(
        hasButtons: Boolean,
        hasExactAlarmPermission: Boolean
    ): Int {
        sdkInstance.logger.log { "$tag getDecoratedStyleExpandedProgressbarLayoutId() : hasButton: $hasButtons, hasExactAlarmPermission: $hasExactAlarmPermission" }
        return if (hasButtons) {
            if (hasExactAlarmPermission) {
                R.layout.moe_rich_push_progressbar_expanded_with_action_buttons_decorated_style
            } else {
                R.layout.moe_rich_push_timer_with_title_expanded_with_action_buttons_decorated_style
            }
        } else {
            if (hasExactAlarmPermission) {
                R.layout.moe_rich_push_progressbar_expanded_without_action_buttons_decorated_style
            } else {
                R.layout.moe_rich_push_timer_with_title_expanded_without_action_buttons_decorated_style
            }
        }
    }
}