/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.richnotification.internal.repository

import android.os.Build
import android.widget.ImageView
import androidx.annotation.Nullable
import androidx.annotation.VisibleForTesting
import com.moengage.core.LogLevel
import com.moengage.core.internal.logger.Logger
import com.moengage.pushbase.internal.NOTIFICATION_MESSAGE
import com.moengage.pushbase.internal.NOTIFICATION_SUMMARY
import com.moengage.pushbase.internal.NOTIFICATION_TITLE
import com.moengage.pushbase.internal.PAYLOAD_ATTRIBUTE_RICH_PUSH
import com.moengage.pushbase.internal.repository.ActionParser
import com.moengage.pushbase.model.action.Action
import com.moengage.richnotification.internal.ASSET_COLOR_LIGHT_GREY
import com.moengage.richnotification.internal.DEFAULT_DISMISS_CTA_TEXT
import com.moengage.richnotification.internal.DEFAULT_HEADER_STATE_FOR_BANNER_TEMPLATE
import com.moengage.richnotification.internal.EXPANDED_IMAGE_TEXT_BANNER
import com.moengage.richnotification.internal.MODULE_TAG
import com.moengage.richnotification.internal.PROPERTY_DURATION_KEY
import com.moengage.richnotification.internal.PROPERTY_EXPIRY_KEY
import com.moengage.richnotification.internal.PROPERTY_FORMAT_KEY
import com.moengage.richnotification.internal.TEMPLATE_NAME
import com.moengage.richnotification.internal.TEMPLATE_NAME_IMAGE_BANNER
import com.moengage.richnotification.internal.TEMPLATE_NAME_TIMER
import com.moengage.richnotification.internal.TEMPLATE_NAME_TIMER_WITH_PROGRESS
import com.moengage.richnotification.internal.WIDGET_TYPE_IMAGE
import com.moengage.richnotification.internal.WIDGET_TYPE_PROGRESSBAR
import com.moengage.richnotification.internal.WIDGET_TYPE_TIMER
import com.moengage.richnotification.internal.models.Card
import com.moengage.richnotification.internal.models.ChronometerProperties
import com.moengage.richnotification.internal.models.ChronometerStyle
import com.moengage.richnotification.internal.models.ChronometerWidget
import com.moengage.richnotification.internal.models.CollapsedBannerTemplate
import com.moengage.richnotification.internal.models.CollapsedTemplate
import com.moengage.richnotification.internal.models.DefaultText
import com.moengage.richnotification.internal.models.DismissCta
import com.moengage.richnotification.internal.models.ExpandedBannerTemplate
import com.moengage.richnotification.internal.models.ExpandedTemplate
import com.moengage.richnotification.internal.models.HeaderStyle
import com.moengage.richnotification.internal.models.ImageWidget
import com.moengage.richnotification.internal.models.LayoutStyle
import com.moengage.richnotification.internal.models.ProgressbarProperties
import com.moengage.richnotification.internal.models.ProgressbarWidget
import com.moengage.richnotification.internal.models.Style
import com.moengage.richnotification.internal.models.Template
import com.moengage.richnotification.internal.models.TimerProperties
import com.moengage.richnotification.internal.models.TimerTemplate
import com.moengage.richnotification.internal.models.Widget
import com.moengage.richnotification.internal.models.WidgetProperties
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject

/**
 * @author Umang Chamaria
 * Date: 02/03/20
 */
private const val tag = "${MODULE_TAG}PayloadParser"
private const val ANDROID_SPECIFIC_KEYS = "android"
private const val INDICATOR_COLOR = "indicatorColor"
private const val SHOULD_SHOW_LARGE_ICON = "showLargeIcon"

private const val DEFAULT_ACTION = "defaultActions"
private const val COLLAPSED_STATE = "collapsed"
private const val EXPANDED_STATE = "expanded"
private const val TYPE = "type"
private const val ID = "id"
private const val CONTENT = "content"
private const val STYLE = "style"
private const val BACKGROUND_COLOR = "bgColor"
private const val WIDGET_PROPERTIES = "prop"
private const val PROPERTIES = "props"

// this is to fetch the duration and expiry for timer and timerWithProgress bar required for
// TimerTemplate
private const val PROPERTY_1 = "prop1"
private const val TEXT_COLOR = "color"

// expanded state
private const val WIDGETS = "widgets"
private const val ACTIONS = "actions"
internal const val CARDS = "cards"
private const val ACTION_BUTTONS = "actionButton"
private const val AUTO_START = "autoStart"
private const val NAME = "name"
private const val VALUE = "value"
private const val VALUE_OF = "valueOf"
private const val KV_PAIR = "kvPairs"
private const val MESSAGE = "message"

// remind me later
private const val REMIND_AFTER = "remindAfterHours"
private const val TOMORROW_AT = "remindTomorrowAt"

private const val APP_NAME_COLOR = "appNameColor"

private const val SHOW_HEADER = "showHeader"
private const val DISMISS_CTA = "dismissCta"
private const val SCALE_TYPE = "scaleType"
private const val SCALE_TYPE_CENTER_CROP = "cc"
private const val SCALE_TYPE_CENTER_INSIDE = "ci"

internal class PayloadParser {

    @Nullable
    fun parseTemplate(payloadString: String): Template? {
        try {
            val payloadJson = JSONObject(payloadString)
            if (payloadJson.length() == 0 || !payloadJson.has(PAYLOAD_ATTRIBUTE_RICH_PUSH)) return null
            val richPushJson = payloadJson.getJSONObject(PAYLOAD_ATTRIBUTE_RICH_PUSH) ?: return null
            val templateType = getTemplateType(richPushJson)
            return if (templateType == TEMPLATE_NAME_TIMER) {
                getTimerTemplate(richPushJson)
            } else {
                getBaseTemplate(richPushJson)
            }
        } catch (t: Throwable) {
            Logger.print(LogLevel.ERROR, t) { "$tag parseTemplate() : " }
        }
        return null
    }

    @Throws(JSONException::class)
    private fun defaultTextFromJson(richPushJson: JSONObject): DefaultText {
        return DefaultText(
            richPushJson.optString(NOTIFICATION_TITLE, ""),
            richPushJson.optString(NOTIFICATION_MESSAGE, ""),
            richPushJson.optString(NOTIFICATION_SUMMARY, "")
        )
    }

    @Throws(JSONException::class)
    private fun parseCollapsedTemplate(richPushJson: JSONObject): CollapsedTemplate? {
        if (!richPushJson.has(COLLAPSED_STATE)) return null
        val collapsedJson = richPushJson.getJSONObject(COLLAPSED_STATE)
        val templateType = collapsedJson.getString(TYPE) ?: return null
        return when (templateType) {
            EXPANDED_IMAGE_TEXT_BANNER, TEMPLATE_NAME_IMAGE_BANNER -> {
                collapsedBannerTemplateFromJson(collapsedJson, richPushJson)
            }
            else -> {
                baseCollapsedTemplateFromJson(collapsedJson, richPushJson)
            }
        }
    }

    @Throws(JSONException::class)
    private fun parseExpandedTemplate(richPushJson: JSONObject): ExpandedTemplate? {
        if (!richPushJson.has(EXPANDED_STATE)) return null
        val expandedState = richPushJson.getJSONObject(EXPANDED_STATE)
        val templateType = expandedState.getString(TYPE) ?: return null

        return when (templateType) {
            EXPANDED_IMAGE_TEXT_BANNER, TEMPLATE_NAME_IMAGE_BANNER -> {
                expandedBannerTemplateFromJson(expandedState, richPushJson)
            }
            else -> {
                baseExpandedTemplateFromJson(expandedState, richPushJson)
            }
        }
    }

    @Throws(JSONException::class)
    private fun actionButtonListFromJson(
        expandedState: JSONObject,
        richPushJson: JSONObject
    ): List<Widget> {
        if (!expandedState.has(ACTION_BUTTONS)) return emptyList<Widget>()
        val actionButtons = expandedState.getJSONArray(ACTION_BUTTONS)
        return if (actionButtons == null || actionButtons.length() == 0) {
            emptyList<Widget>()
        } else {
            widgetListFromJson(
                actionButtons,
                richPushJson
            )
        }
    }

    @Throws(JSONException::class)
    private fun cardListFromJson(
        expandedState: JSONObject,
        richPushJson: JSONObject
    ): MutableList<Card> {
        if (!expandedState.has(CARDS)) return emptyList<Card>().toMutableList()
        val cardArray = expandedState.getJSONArray(CARDS)
        val cards: MutableList<Card> = ArrayList(cardArray.length())
        for (i in 0 until cardArray.length()) {
            val cardJson = cardArray.getJSONObject(i)
            cards.add(cardFromJson(cardJson, richPushJson))
        }
        return cards
    }

    @Throws(JSONException::class)
    private fun cardFromJson(cardJson: JSONObject, richPushJson: JSONObject): Card {
        return Card(
            cardJson.getInt(ID),
            widgetListFromJson(cardJson.getJSONArray(WIDGETS), richPushJson),
            cardJson.getString(TYPE),
            if (cardJson.has(ACTIONS)) {
                actionListFromJson(cardJson.getJSONArray(ACTIONS))
            } else {
                emptyArray<Action>()
            }
        )
    }

    @Throws(JSONException::class)
    private fun widgetListFromJson(
        widgetsArray: JSONArray,
        richPushJson: JSONObject
    ): List<Widget> {
        val widgetList: MutableList<Widget> = ArrayList(widgetsArray.length())
        for (i in 0 until widgetsArray.length()) {
            val widgetJson = widgetsArray.getJSONObject(i)
            val widget = widgetFromJson(widgetJson, richPushJson)
            if (widget != null) {
                widgetList.add(widget)
            }
        }
        return widgetList
    }

    @Throws(JSONException::class)
    private fun baseWidgetFromJson(widgetJson: JSONObject, widgetType: String): Widget {
        return Widget(
            widgetType,
            widgetJson.getInt(ID),
            if (widgetType == WIDGET_TYPE_TIMER || widgetType == WIDGET_TYPE_PROGRESSBAR) {
                ""
            } else {
                widgetJson.getString(
                    CONTENT
                )
            },
            if (widgetJson.has(STYLE)) {
                styleFromJson(
                    widgetJson.getJSONObject(STYLE),
                    widgetType
                )
            } else {
                null
            },
            if (widgetJson.has(ACTIONS)) {
                actionListFromJson(
                    widgetJson.getJSONArray(ACTIONS)
                )
            } else {
                emptyArray()
            }
        )
    }

    @Throws(JSONException::class)
    private fun styleFromJson(styleJson: JSONObject, widgetType: String): Style {
        return if (widgetType == WIDGET_TYPE_TIMER) {
            chronometerStyleFromJson(styleJson)
        } else {
            Style(styleJson.getString(BACKGROUND_COLOR))
        }
    }

    @Throws(JSONException::class)
    private fun actionListFromJson(actionArray: JSONArray): Array<Action> {
        val actionList: MutableList<Action> = ArrayList(actionArray.length())
        val actionParser = ActionParser()
        for (i in 0 until actionArray.length()) {
            val action = actionParser.actionFromJson(actionArray.getJSONObject(i))
            if (action != null) {
                actionList.add(action)
            }
        }
        return actionList.toTypedArray()
    }

    @Throws(JSONException::class)
    private fun layoutStyleFromJson(collapsedState: JSONObject): LayoutStyle? {
        if (!collapsedState.has(STYLE)) return null
        return if (!collapsedState.getJSONObject(STYLE)
            .has(BACKGROUND_COLOR)
        ) {
            null
        } else {
            LayoutStyle(collapsedState.getJSONObject(STYLE).getString(BACKGROUND_COLOR))
        }
    }

    @Throws(JSONException::class)
    private fun headerStyleFromJson(richPushJson: JSONObject): HeaderStyle {
        if (!richPushJson.has(APP_NAME_COLOR)) return HeaderStyle(null)
        return HeaderStyle(richPushJson.getString(APP_NAME_COLOR))
    }

    private fun baseExpandedTemplateFromJson(
        expandedJson: JSONObject,
        richPushJson: JSONObject
    ): ExpandedTemplate {
        // Auto start is not supported from Android version 11, so make it manual by default
        val autostart = if (Build.VERSION.SDK_INT > Build.VERSION_CODES.Q) {
            false
        } else {
            expandedJson.optBoolean(AUTO_START, false)
        }

        return ExpandedTemplate(
            expandedJson.getString(TYPE),
            layoutStyleFromJson(expandedJson),
            actionButtonListFromJson(expandedJson, richPushJson),
            cardListFromJson(expandedJson, richPushJson),
            autostart
        )
    }

    fun expandedBannerTemplateFromJson(
        expandedJson: JSONObject,
        richPushJson: JSONObject
    ): ExpandedBannerTemplate {
        return ExpandedBannerTemplate(
            baseExpandedTemplateFromJson(expandedJson, richPushJson),
            expandedJson.optBoolean(SHOW_HEADER, DEFAULT_HEADER_STATE_FOR_BANNER_TEMPLATE)
        )
    }

    private fun baseCollapsedTemplateFromJson(
        collapsedJson: JSONObject,
        richPushJson: JSONObject
    ): CollapsedTemplate {
        return CollapsedTemplate(
            collapsedJson.getString(TYPE),
            layoutStyleFromJson(collapsedJson),
            cardListFromJson(collapsedJson, richPushJson)
        )
    }

    fun collapsedBannerTemplateFromJson(
        collapsedJson: JSONObject,
        richPushJson: JSONObject
    ): CollapsedBannerTemplate {
        return CollapsedBannerTemplate(
            baseCollapsedTemplateFromJson(collapsedJson, richPushJson),
            collapsedJson.optBoolean(SHOW_HEADER, DEFAULT_HEADER_STATE_FOR_BANNER_TEMPLATE)
        )
    }

    @Throws(JSONException::class)
    private fun getJsonFromReferencePath(
        responseJson: JSONObject,
        contentPath: String
    ): JSONObject {
        val pathPieces = contentPath.split("/").toTypedArray()
        var propertiesPayload = responseJson
        for (i in 1 until pathPieces.size)
            propertiesPayload = propertiesPayload.getJSONObject(pathPieces[i])
        return propertiesPayload
    }

    @Throws(JSONException::class)
    private fun getBaseTemplate(richPushJson: JSONObject): Template {
        return Template(
            richPushJson.getString(TEMPLATE_NAME),
            defaultTextFromJson(richPushJson),
            actionListFromJson(richPushJson.getJSONArray(DEFAULT_ACTION)),
            parseCollapsedTemplate(richPushJson),
            parseExpandedTemplate(richPushJson),
            richPushJson.getJSONObject(ANDROID_SPECIFIC_KEYS)
                .optString(INDICATOR_COLOR, ASSET_COLOR_LIGHT_GREY),
            richPushJson.getJSONObject(ANDROID_SPECIFIC_KEYS).getBoolean
            (SHOULD_SHOW_LARGE_ICON),
            headerStyleFromJson(richPushJson),
            getDismissCtaFromJson(richPushJson)
        )
    }

    @Throws(JSONException::class)
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun chronometerPropertiesFromJson(
        richPushJson: JSONObject,
        propertiesPath: String
    ): ChronometerProperties {
        val properties = getJsonFromReferencePath(richPushJson, propertiesPath)
        return ChronometerProperties(
            properties.getLong(PROPERTY_DURATION_KEY),
            properties.getLong(PROPERTY_EXPIRY_KEY),
            properties.getString(PROPERTY_FORMAT_KEY),
            WidgetProperties(properties)
        )
    }

    @Throws(JSONException::class)
    private fun getTemplateType(richPushJson: JSONObject): String {
        if (!richPushJson.has(COLLAPSED_STATE)) return ""
        val collapsedJson = richPushJson.getJSONObject(COLLAPSED_STATE)
        val templateType = collapsedJson.getString(TYPE) ?: return ""
        return if (templateType == TEMPLATE_NAME_TIMER ||
            templateType == TEMPLATE_NAME_TIMER_WITH_PROGRESS
        ) {
            TEMPLATE_NAME_TIMER
        } else {
            ""
        }
    }

    @Throws(JSONException::class)
    private fun getTimerTemplate(richPushJson: JSONObject): TimerTemplate {
        val baseTemplate = getBaseTemplate(richPushJson)
        return TimerTemplate(
            baseTemplate,
            timerPropertiesFromBaseTemplate(baseTemplate)
        )
    }

    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    @Throws(JSONException::class)
    fun chronometerStyleFromJson(styleJson: JSONObject): ChronometerStyle {
        return ChronometerStyle(styleJson.getString(TEXT_COLOR))
    }

    @Throws(JSONException::class)
    private fun widgetFromJson(
        widgetJson: JSONObject,
        richPushJson: JSONObject
    ): Widget {
        return when (val widgetType = widgetJson.getString(TYPE)) {
            WIDGET_TYPE_TIMER -> chronometerWidgetFromJson(widgetJson, richPushJson)
            WIDGET_TYPE_PROGRESSBAR -> progressbarWidgetFromJson(widgetJson, richPushJson)
            WIDGET_TYPE_IMAGE -> imageWidgetFromJson(widgetJson, widgetType)
            else -> baseWidgetFromJson(widgetJson, widgetType)
        }
    }

    @Throws(JSONException::class)
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun chronometerWidgetFromJson(
        widgetJson: JSONObject,
        richPushJson: JSONObject
    ): ChronometerWidget {
        return ChronometerWidget(
            baseWidgetFromJson(widgetJson, WIDGET_TYPE_TIMER),
            chronometerPropertiesFromJson(richPushJson, widgetJson.getString(WIDGET_PROPERTIES))
        )
    }

    @Throws(JSONException::class)
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun timerPropertiesFromBaseTemplate(baseTemplate: Template): TimerProperties {
        var timerProperties: TimerProperties? = null

        if (baseTemplate.collapsedTemplate != null &&
            baseTemplate.collapsedTemplate.cards.isNotEmpty()
        ) {
            timerProperties =
                getTimerPropertiesFromWidgetList(baseTemplate.collapsedTemplate.cards[0].widgets)
        }

        if (timerProperties == null && baseTemplate.expandedTemplate != null &&
            baseTemplate.expandedTemplate.cards.isNotEmpty()
        ) {
            timerProperties =
                getTimerPropertiesFromWidgetList(baseTemplate.expandedTemplate.cards[0].widgets)
        }

        return timerProperties ?: TimerProperties(-1L, -1L)
    }

    @Throws(JSONException::class)
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun progressbarWidgetFromJson(
        widgetJson: JSONObject,
        richPushJson: JSONObject
    ): ProgressbarWidget {
        return ProgressbarWidget(
            baseWidgetFromJson(widgetJson, WIDGET_TYPE_PROGRESSBAR),
            progressbarPropertiesFromJson(richPushJson, widgetJson.getString(WIDGET_PROPERTIES))
        )
    }

    @Throws(JSONException::class)
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    fun progressbarPropertiesFromJson(
        richPushJson: JSONObject,
        propertiesPath: String
    ): ProgressbarProperties {
        val properties = getJsonFromReferencePath(richPushJson, propertiesPath)
        return ProgressbarProperties(
            properties.getLong(PROPERTY_DURATION_KEY),
            properties.getLong(PROPERTY_EXPIRY_KEY),
            WidgetProperties(properties)
        )
    }

    private fun getTimerPropertiesFromWidgetList(widgetList: List<Widget>): TimerProperties? {
        for (widget in widgetList) {
            if (widget is ChronometerWidget) {
                return TimerProperties(
                    widget.properties.duration,
                    widget.properties.expiry
                )
            }
            if (widget is ProgressbarWidget) {
                return TimerProperties(
                    widget.properties.duration,
                    widget.properties.expiry
                )
            }
        }
        return null
    }

    private fun getDismissCtaFromJson(richPushJson: JSONObject): DismissCta {
        return DismissCta(richPushJson.optString(DISMISS_CTA, DEFAULT_DISMISS_CTA_TEXT))
    }

    private fun imageWidgetFromJson(widgetJson: JSONObject, widgetType: String): ImageWidget {
        return ImageWidget(
            baseWidgetFromJson(widgetJson, widgetType),
            getImageWidgetScaleType(widgetJson)
        )
    }

    private fun getImageWidgetScaleType(widgetJson: JSONObject): ImageView.ScaleType {
        return when (widgetJson.optString(SCALE_TYPE, "")) {
            SCALE_TYPE_CENTER_CROP -> ImageView.ScaleType.CENTER_CROP
            SCALE_TYPE_CENTER_INSIDE -> ImageView.ScaleType.CENTER_INSIDE
            else -> ImageView.ScaleType.CENTER_CROP
        }
    }
}