package com.moloco.sdk.acm

import android.content.Context
import androidx.annotation.VisibleForTesting
import androidx.lifecycle.ProcessLifecycleOwner
import com.moloco.sdk.acm.db.MetricsDb
import com.moloco.sdk.acm.eventprocessing.DBWorkRequestImpl
import com.moloco.sdk.acm.eventprocessing.EventProcessor
import com.moloco.sdk.acm.eventprocessing.EventProcessorImpl
import com.moloco.sdk.acm.eventprocessing.RequestScheduler
import com.moloco.sdk.acm.eventprocessing.RequestSchedulerTimer
import com.moloco.sdk.acm.services.ApplicationLifecycleTrackerImpl
import com.moloco.sdk.acm.services.MolocoMetricsLogger
import com.moloco.sdk.acm.services.ApplicationLifecycleObserver
import com.moloco.sdk.acm.services.TimeProviderServiceImpl
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import java.lang.IllegalStateException
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.atomic.AtomicReference

/**
 * Callback interface for initialization of the AndroidClientMetrics Library.
 */
interface AndroidClientMetricsCallback {
    /**
     * Called when initialization of AndroidClientMetrics is successful.
     */
    fun onInitializationSuccess()

    /**
     * Called when initialization of AndroidClientMetrics fails.
     * @param e The exception that caused the initialization failure.
     */
    fun onInitializationFailure(e: Exception)
}

/**
 * Enum for giving us the initialization status of the Android Client Metrics library
 */
internal enum class InitializationStatus {
    /**
     * Library is initialized
      */
    INITIALIZED,
    /**
     * Library is initializing
     */
    INITIALIZING,
    /**
     * Library is not initialized
     */
    UNINITIALIZED
}

/**
 * Object responsible for handling analytics events for Moloco SDK.
 */
object AndroidClientMetrics {
    private lateinit var eventProcessor: EventProcessor
    private lateinit var applicationLifecycleTracker: ApplicationLifecycleTrackerImpl
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    internal lateinit var opsConfig: ACMConfig
    private var pendingConfigUpdate: UpdateConfig? = null
    private val configMutex = Mutex()
    private val ioScope = CoroutineScope(Dispatchers.IO + SupervisorJob())

    private val _initializationStatus = AtomicReference(InitializationStatus.UNINITIALIZED)
    // Events stored before initialization and to be process once ACM has initialized
    private val preInitTimerList = CopyOnWriteArrayList<TimerEvent>()
    private val preInitCountList = CopyOnWriteArrayList<CountEvent>()
    private lateinit var requestScheduler: RequestScheduler
    private const val TAG = "AndroidClientMetrics"

    /**
     * Property that returns initialization state of the AndroidClientMetrics Library
     */
    internal val initializationStatus: InitializationStatus
        get() = _initializationStatus.get()

    /**
     * Initializes the AndroidClientMetrics with the given configuration.
     *
     * @param config The configuration used to initialize the analytics system.
     */
    fun initialize(config: InitConfig, callback: AndroidClientMetricsCallback? = null) {
        MolocoMetricsLogger.info(TAG, "ACM initialize")
        if (_initializationStatus.compareAndSet(InitializationStatus.UNINITIALIZED, InitializationStatus.INITIALIZING)) {
            opsConfig = config.toACMConfig()
            ioScope.launch {
                try {
                    val metricsDAO = MetricsDb.getInstance(config.context).operationalMetricsDao()
                    val timeProviderService = TimeProviderServiceImpl()
                    val dbWorkRequest = DBWorkRequestImpl(opsConfig, config.context)

                    requestScheduler = RequestSchedulerTimer(
                        dbWorkRequest = dbWorkRequest,
                        opsConfig = opsConfig,
                        coroutineScope = ioScope
                    )

                    applicationLifecycleTracker = ApplicationLifecycleTrackerImpl(
                        ProcessLifecycleOwner.get().lifecycle,
                        ApplicationLifecycleObserver(dbWorkRequest, ioScope)
                    )

                    eventProcessor = EventProcessorImpl(
                        metricsDAO,
                        timeProviderService,
                        requestScheduler,
                        applicationLifecycleTracker
                    )

                    _initializationStatus.set(InitializationStatus.INITIALIZED)
                    configMutex.withLock {
                        pendingConfigUpdate?.let {configUpdate ->
                            pendingConfigUpdate = null
                            MolocoMetricsLogger.debug(TAG, "Updating config with pending config")
                            updateConfigInternal(configUpdate)
                        }
                    }

                    processQueuedEvents()
                    callback?.onInitializationSuccess()
                } catch (e: IllegalStateException) {
                    MolocoMetricsLogger.error(MetricsDb.TAG, "Unable to create metrics db", e)
                    _initializationStatus.set(InitializationStatus.UNINITIALIZED)
                    callback?.onInitializationFailure(e)
                } catch (e: Exception) {
                    MolocoMetricsLogger.error(TAG, "Initialization error", e)
                    _initializationStatus.set(InitializationStatus.UNINITIALIZED)
                    callback?.onInitializationFailure(e)
                }
            }
        }
    }

    /**
     * Updates the configuration of the AndroidClientMetrics. ACM library
     * can be run as soon as it is initialized. However it is possible for consuming application / SDK
     * to want to fetch configurations asynchronously and update the ACM library with the new configurations.
     *
     * Caller should make sure that ACM initialize is called at least once for updateConfig effect to take place
     */
    suspend fun updateConfig(newConfig: UpdateConfig) {
        // If the SDK is still not initialized, then we cannot update the config
        // In reality this shouldn't happen. This is just a safety check. The SDK should be initialized by now but if not
        // then we keep a pending config update and apply it once the SDK is initialized
        if (_initializationStatus.get() != InitializationStatus.INITIALIZED) {
            MolocoMetricsLogger.warn(TAG, "ACM updateConfig called when the SDK was not initialized. Initialize the SDK first.")
            configMutex.withLock {
                pendingConfigUpdate = newConfig
            }

            return
        }

        MolocoMetricsLogger.info(TAG, "ACM update called. ACM initialized already, proceeding with update")
        updateConfigInternal(newConfig)
    }

    private suspend fun updateConfigInternal(newConfig: UpdateConfig) {
        newConfig.postAnalyticsUrl?.let {
            opsConfig.postAnalyticsUrl = it
        }

        newConfig.requestPeriodSeconds?.let {
            opsConfig.requestPeriodSeconds = it
        }
        requestScheduler.resetScheduleAndTriggerNewScheduledUpload()
    }

    /**
     * Records a count event.
     *
     * @param event The operational count event to be recorded.
     */
    fun recordCountEvent(event: CountEvent) {
        if (_initializationStatus.get() != InitializationStatus.INITIALIZED) {
            preInitCountList.add(event)
            MolocoMetricsLogger.debug(TAG, "Moloco Client Metrics not initialized")
            return
        }
        ioScope.launch {
            eventProcessor.processCountEvent(event)
        }
    }

    /**
     * Starts a timer event for analytics tracking.
     *
     * This function initializes an `AnalyticsTimerEvent` with the provided `eventName`
     * and starts the timer for tracking the event's duration.
     * We can return an event regardless if the Library is initialized or not.
     *
     * @param eventName the name of the event to track.
     * @return an instance of `AnalyticsTimerEvent` with the timer started.
     *
     */
    fun startTimerEvent(eventName: String): TimerEvent {
        if (_initializationStatus.get() != InitializationStatus.INITIALIZED) {
            MolocoMetricsLogger.debug(TAG, "Moloco Client Metrics not initialized")
        }
        return TimerEvent.create(eventName).apply { startTimer() }
    }

    /**
     * Records a timer event.
     *
     * @param event The operational timer event to be recorded.
     */
    fun recordTimerEvent(event: TimerEvent) {
        event.stopTimer()
        if (_initializationStatus.get() != InitializationStatus.INITIALIZED) {
            preInitTimerList.add(event)
            MolocoMetricsLogger.debug(TAG, "Moloco Client Metrics not initialized")
            return
        }
        ioScope.launch {
            eventProcessor.processTimerEvent(event)
        }
    }

    /**
     * Processes events that were stored before the ACM library was initialized
     */
    private fun processQueuedEvents() {
        ioScope.launch {
            preInitTimerList.forEach { eventProcessor.processTimerEvent(it) }
            preInitCountList.forEach { eventProcessor.processCountEvent(it) }
            preInitTimerList.clear()
            preInitCountList.clear()
        }
    }

    /**
     * Triggers an app background.
     * We will only use this specifically for instrumentation tests.
     */
    @VisibleForTesting
    internal fun triggerBackgroundEvent() {
        if (this::applicationLifecycleTracker.isInitialized) {
            applicationLifecycleTracker.triggerBackgroundEvent()
        }
    }
}
