package com.moloco.sdk.acm

import com.moloco.sdk.acm.services.Stopwatch
import com.moloco.sdk.acm.services.TimeProviderServiceImpl

/**
 * Represents an analytics event that measures the time taken for a specific operation.
 *
 * @param eventName The name of the event.
 */
class TimerEvent private constructor(eventName: String,
                                     private val stopwatch: Stopwatch): Event {

    private var timeInMillis: Long = 0L

    companion object {
        /**
         * The maximum number of tags allowed per event.
         */
        private const val MAX_TAGS_PER_EVENT = 10

        /**
         * The maximum character length allowed for a tag.
         */
        private const val MAX_TAG_CHAR_LEN = 50

        /**
         * Factory method to create an instance of [TimerEvent].
         *
         * @param eventName The name of the event.
         */
        fun create(eventName: String): TimerEvent {
            val stopwatch = Stopwatch(TimeProviderServiceImpl())
            return TimerEvent(eventName, stopwatch)
        }
    }

    /**
     * The list of tags associated with the event.
     */
    override val eventTags = mutableListOf<EventTag>()

    /**
     * The name of the event.
     */
    override val name = eventName

    /**
     * The amount time the event took to complete.
     */
    val time: Long
        get() = timeInMillis

    /**
     * Starts the timer by invoking the start method of the internal stopwatch.
     */
    fun startTimer() {
        stopwatch.start()
    }

    /**
     * Stops the timer and calculates the elapsed time by invoking the end method of the internal stopwatch.
     * The elapsed time is stored in the [timeInMillis] property.
     */
    fun stopTimer() {
        if (timeInMillis == 0L) timeInMillis = stopwatch.end()
    }

    /**
     * Sets the timer value for the TimerEvent.
     *
     * @param time The timer value to be set. It must be non-negative.
     * @return The modified TimerEvent object with the timer value set.
     * @throws IllegalArgumentException If the provided value is negative.
     */
    fun withTime(time: Long): TimerEvent {
        require(time >= 0) { "Count cannot be negative" }
        return this.apply { timeInMillis = time }
    }

    /**
     * Adds a tag to the operational timer event.
     *
     * @param key The key of the tag.
     * @param value The value of the tag.
     * @return The updated OperationalTimerEvent with the new tag.
     */
    override fun withTag(key: String, value: String): TimerEvent {
        if (eventTags.size >= MAX_TAGS_PER_EVENT
            || key.length > MAX_TAG_CHAR_LEN
            || value.length > MAX_TAG_CHAR_LEN
        ) {

            if (BuildConfig.DEBUG) {
                throw IllegalArgumentException("Exceeded tag constraints: MAX_TAGS_PER_EVENT = $MAX_TAGS_PER_EVENT, MAX_TAG_CHAR_LEN = $MAX_TAG_CHAR_LEN")
            }

            return this
        }
        eventTags.add(EventTag(key, value))
        return this
    }
}
