package com.moloco.sdk.acm.db

import androidx.room.Dao
import androidx.room.Insert
import androidx.room.OnConflictStrategy
import androidx.room.Query
import androidx.room.Transaction

@Dao
internal interface MetricsDAO {
    @Insert(onConflict = OnConflictStrategy.REPLACE)
    fun insertEvent(event: EventEntity): Long

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    fun insertAll(events: List<EventEntity>): List<Long>

    // Insert without overwriting existing rows
    @Insert(onConflict = OnConflictStrategy.ABORT)
    fun insertAllWithUniqueIds(events: List<EventEntity>): List<Long>

    @Query("SELECT * FROM events WHERE id = :eventId")
    fun getEventById(eventId: Long): EventEntity?

    @Query("SELECT * FROM events")
    fun getAllEvents(): List<EventEntity>

    /**
     * This method queries events with a limit - sqlite variable limit is 999
     * Chose to be under the limit to be safe.
     */
    @Query("SELECT * FROM events LIMIT 900")
    fun getAllEventsLimit(): List<EventEntity>

    @Query("DELETE FROM events")
    suspend fun deleteAllEvents()

    @Query("DELETE FROM sqlite_sequence WHERE name='events'")
    suspend fun resetEventAutoIncrement()

    @Query("DELETE FROM events WHERE id IN (:ids)")
    suspend fun deleteEventsByIds(ids: List<Long>)

    // TODO: Write query to allow for data to be added back
    /**
     * This method chunks deletions/queries - sqlite variable limit is 999
     */
    @Transaction
    suspend fun deleteAndReturnDeletedEvents(): List<EventEntity> {
        val deletedEvents = mutableListOf<EventEntity>()
        while (true) {
            val eventsToDeleteChunk = getAllEventsLimit()
            if (eventsToDeleteChunk.isEmpty()) {
                break
            }

            val idsToDelete = eventsToDeleteChunk.map { it.id }

            deleteEventsByIds(idsToDelete)
            deletedEvents.addAll(eventsToDeleteChunk)
        }

        resetEventAutoIncrement()

        return deletedEvents
    }

    @Transaction
    suspend fun resetDatabase() {
        deleteAllEvents()
        resetEventAutoIncrement()
    }
}
