package com.moloco.sdk.acm.eventprocessing

import android.content.Context
import android.util.Log
import androidx.work.BackoffPolicy
import androidx.work.Constraints
import androidx.work.CoroutineWorker
import androidx.work.Data
import androidx.work.NetworkType
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.WorkManager
import androidx.work.WorkRequest
import androidx.work.WorkerParameters
import androidx.work.workDataOf
import com.moloco.sdk.acm.ACMConfig
import com.moloco.sdk.acm.AcmHeaders.APP_BUNDLE
import com.moloco.sdk.acm.AcmHeaders.APP_KEY
import com.moloco.sdk.acm.AcmHeaders.APP_VERSION
import com.moloco.sdk.acm.AcmHeaders.MEDIATOR
import com.moloco.sdk.acm.AcmHeaders.MOLOCO_SDK_VERSION
import com.moloco.sdk.acm.AcmHeaders.OS
import com.moloco.sdk.acm.AcmHeaders.OS_VERSION
import com.moloco.sdk.acm.db.MetricsDb
import com.moloco.sdk.acm.eventprocessing.DefaultDataTimeConfigurations.DATA_AGE_SECONDS
import com.moloco.sdk.acm.http.AcmHeadersBuilder
import com.moloco.sdk.acm.http.HttpClientSingleton
import com.moloco.sdk.acm.http.metricsHttpClient
import com.moloco.sdk.acm.services.MolocoMetricsLogger
import com.moloco.sdk.acm.services.TimeProviderServiceImpl
import java.util.concurrent.TimeUnit

/**
 * Interface for encapsulating a WorkRequest for workRequest that manages
 * data upload and purge operations.
 */
internal interface DBWorkRequest {
    /**
     * Runs an upload of data and purges old data from our DB.
     * This method is responsible for initiating the upload process and
     * ensuring that outdated data is removed.
     */
    fun uploadAndPurge()
}

/**
 * Class representing the RequestScheduler for managing data upload and purge operations.
 */
internal class DBWorkRequestImpl(private val config: ACMConfig, private val context: Context) : DBWorkRequest {

    override fun uploadAndPurge() {
        val constraints = Constraints.Builder()
            .setRequiredNetworkType(NetworkType.CONNECTED)
            .build()

        val dataMap = mapOf(
            "url" to config.postAnalyticsUrl,
            APP_KEY to config.clientOptions[APP_KEY],
            APP_BUNDLE to config.clientOptions[APP_BUNDLE],
            APP_VERSION to config.clientOptions[APP_VERSION],
            OS to config.clientOptions[OS],
            OS_VERSION to config.clientOptions[OS_VERSION],
            MOLOCO_SDK_VERSION to config.clientOptions[MOLOCO_SDK_VERSION],
            MEDIATOR to config.clientOptions[MEDIATOR]
        )

        val data = dataMap.getAsData() ?: return

        // TODO: backoff policy here or in HTTP layer - and configurable?
        val workRequest = OneTimeWorkRequestBuilder<DBRequestWorker>()
            .setConstraints(constraints)
            .setInputData(data)
            .setBackoffCriteria(
                BackoffPolicy.EXPONENTIAL,
                WorkRequest.MIN_BACKOFF_MILLIS,
                TimeUnit.MILLISECONDS
            ).build()

        WorkManager.getInstance(context).enqueue(workRequest)
    }
}

internal class DBRequestWorker(context: Context, params: WorkerParameters)
    : CoroutineWorker(context, params) {
    private val TAG = "DBRequestWorker"
    private val metricsDAO = MetricsDb.getInstance(context).operationalMetricsDao()
    private val url = inputData.getString("url")
    private val clientOptions = mapOf(
        APP_KEY to inputData.getString(APP_KEY),
        APP_BUNDLE to inputData.getString(APP_BUNDLE),
        APP_VERSION to inputData.getString(APP_VERSION),
        OS to inputData.getString(OS),
        OS_VERSION to inputData.getString(OS_VERSION),
        MOLOCO_SDK_VERSION to inputData.getString(MOLOCO_SDK_VERSION),
        MEDIATOR to inputData.getString(MEDIATOR)
    ).filterValues { it != null }

    val headers = AcmHeadersBuilder().createHeaders(
        sdkVersion = clientOptions[MOLOCO_SDK_VERSION],
        androidOSVersion = clientOptions[OS_VERSION],
        key = clientOptions[APP_KEY],
        bundle = clientOptions[APP_BUNDLE],
        appVersion = clientOptions[APP_VERSION],
        mediator = clientOptions[MEDIATOR]
    )

    override suspend fun doWork(): Result {
        return try {
            if (url != null) {
                HttpClientSingleton.initialize(metricsHttpClient(), url)
                RequestAndPurgeDBImpl(
                    HttpClientSingleton.postMetricsRequest,
                    metricsDAO,
                    dataAgeChecker = DataAgeChecker(TimeProviderServiceImpl(), ageSeconds = DATA_AGE_SECONDS),
                    headers
                )()
            }
            Result.success()
        } catch (e: Exception) {
            MolocoMetricsLogger.error(TAG, "Work Manager failure: ${e.message}")
            Result.failure()
        }
    }
}

// Converts the Map to a Data object for Worker
fun Map<String, Any?>.getAsData(): Data? {
    return try {
        workDataOf(*this.map { it.key to it.value }.toTypedArray())
    } catch (e: Exception) {
        Log.e("DBPeriodicRequest", "${e.message}. Data: $this")
        null
    }
}
