package com.moloco.sdk.acm.eventprocessing

import android.database.sqlite.SQLiteException
import android.database.sqlite.SQLiteFullException
import com.moloco.sdk.acm.CountEvent
import com.moloco.sdk.acm.TimerEvent
import com.moloco.sdk.acm.db.EventEntity
import com.moloco.sdk.acm.db.EventType
import com.moloco.sdk.acm.db.MetricsDAO
import com.moloco.sdk.acm.services.ApplicationLifecycleTracker
import com.moloco.sdk.acm.services.MolocoMetricsLogger
import com.moloco.sdk.acm.services.TimeProviderService
import com.moloco.sdk.acm.toKeyValueString
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

/**
 * An interface for internally processing analytics events.
 * This will handle DB syncs and route network requests.
 */
internal interface EventProcessor {
    /**
     * Processes an analytics count event.
     * @param event The count event to process.
     */
    suspend fun processCountEvent(event: CountEvent)

    /**
     * Processes an analytics timer event.
     * @param event The timer event to process.
     */
    suspend fun processTimerEvent(event: TimerEvent)
}

/**
 * A class for internally processing analytics events.
 * This class will handle DB syncs and route requests to a work manager.
 */
internal class EventProcessorImpl(private val metricsDAO: MetricsDAO,
                                  private val timeProviderService: TimeProviderService,
                                  private val requestScheduler: RequestScheduler,
                                  private val applicationLifecycle: ApplicationLifecycleTracker) : EventProcessor {

    /**
     * Processes an analytics count event.
     * @param event The count event to process.
     */
    override suspend fun processCountEvent(event: CountEvent) = withContext(Dispatchers.IO) {
        processEvent(event.name, EventType.COUNT, event.countValue.toLong(), event.eventTags.map { it.toKeyValueString() })
    }

    /**
     * Processes an analytics timer event. It must be a positive number
     * @param event The timer event to process.
     */
    override suspend fun processTimerEvent(event: TimerEvent) = withContext(Dispatchers.IO) {
        if (event.time > 0) {
            processEvent(event.name, EventType.TIMER, event.time, event.eventTags.map { it.toKeyValueString() })
        } else {
            processEvent("negative_time_${event.name}", EventType.TIMER, event.time, event.eventTags.map { it.toKeyValueString() })
        }
    }

    private suspend fun processEvent(name: String, eventType: EventType, data: Long, tags: List<String>) = withContext(Dispatchers.IO) {
        try {
            val eventEntity = EventEntity(
                name = name,
                timestamp = timeProviderService(),
                eventType = eventType,
                data = data,
                tags = tags
            )

            metricsDAO.insertEvent(eventEntity)

            // Internally keeps track of state and only starts schedule on first event
            // If the calling SDK has pending events on app or process restart,
            // they will not be flushed until first/next emitted on restart.
            requestScheduler.scheduleUploadAndPurge()
            applicationLifecycle.startObserving()
        } catch (e: SQLiteException) {
            // In case of improper transaction call or SQLite database full, we will fail gracefully and not record the event
            // The DB will be purged in a fixed time based upon scheduleUploadAndPurge and events for that session can be recorded again
            MolocoMetricsLogger.debug(TAG, "Database error: ${e.message}")
        } catch (e: Exception) {
            MolocoMetricsLogger.debug(TAG, "Unexpected error while processing event: ${e.message}")
        }
    }

    companion object {
        const val TAG = "EventProcessor"
    }
}
