package com.moloco.sdk.acm.eventprocessing

import com.moloco.sdk.acm.ACMConfig
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import java.util.concurrent.Executors
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ScheduledFuture
import java.util.concurrent.TimeUnit

/**
 * Interface representing a scheduler for managing data upload and purge operations.
 */
internal interface RequestScheduler {
    /**
     * Schedules an upload of data and purges old data.
     * This method is responsible for initiating the upload process and
     * ensuring that outdated data is removed.
     *
     * Internally throttles duplicate calls to ensure that only one upload is scheduled at a time.
     */
    suspend fun scheduleUploadAndPurge()

    /**
     * Resets the current scheduled upload of data and triggers a new scheduled upload. Calling this repeatedly
     * will keep canceling the currently scheduled task and reschedule a new one.
     */
    suspend fun resetScheduleAndTriggerNewScheduledUpload()
}

internal class RequestSchedulerTimer(
    private val dbWorkRequest: DBWorkRequest,
    private val opsConfig: ACMConfig,
    private val scheduler: ScheduledExecutorService = Executors.newSingleThreadScheduledExecutor(),
    private val coroutineScope: CoroutineScope
): RequestScheduler {

    private var runningTask: ScheduledFuture<*>? = null
    private val mutex = Mutex()

    override suspend fun scheduleUploadAndPurge() = mutex.withLock {
        schedule()
    }

    override suspend fun resetScheduleAndTriggerNewScheduledUpload() = mutex.withLock {
        runningTask?.cancel(false)
        schedule()
    }

    private fun schedule() {
        if (runningTask == null || runningTask?.isCancelled == true) {
            runningTask = scheduler.scheduleWithFixedDelay({
                coroutineScope.launch {
                    dbWorkRequest.uploadAndPurge()
                }
            }, opsConfig.requestPeriodSeconds, opsConfig.requestPeriodSeconds, TimeUnit.SECONDS)
        }
    }
}
