package com.moloco.sdk.acm.http

import com.moloco.sdk.acm.AcmHeaders.APP_BUNDLE
import com.moloco.sdk.acm.AcmHeaders.APP_KEY
import com.moloco.sdk.acm.AcmHeaders.APP_VERSION
import com.moloco.sdk.acm.AcmHeaders.MEDIATOR
import com.moloco.sdk.acm.AcmHeaders.MOLOCO_SDK_VERSION
import com.moloco.sdk.acm.AcmHeaders.OS
import com.moloco.sdk.acm.AcmHeaders.OS_VERSION
import io.ktor.http.HeadersBuilder

/**
 * Helper class for building headers required for ACM requests.
 */
internal class AcmHeadersBuilder {
    /**
     * Creates headers for ACM requests based on provided parameters.
     *
     * @param sdkVersion The version of the SDK.
     * @param androidOSVersion The version of the Android operating system.
     * @param key The application key.
     * @param bundle The application bundle identifier.
     * @param appVersion The application version.
     * @return Lambda function that configures headers for ACM requests.
     */
    fun createHeaders(
        sdkVersion: String?,
        androidOSVersion: String?,
        key: String?,
        bundle: String?,
        appVersion: String?,
        mediator: String?
    ): HeadersBuilder.() -> Unit = {
        val appInfoHeader = StringBuilder().apply {
            bundle?.let { append("$APP_BUNDLE/$bundle;") }
            appVersion?.let { append("$APP_VERSION/$appVersion;") }
            key?.let { append("$APP_KEY/$key;") }
        }.toString()

        val sdkInfoHeader = StringBuilder().apply {
            sdkVersion?.let { append("$MOLOCO_SDK_VERSION/$sdkVersion;") }
            mediator?.let { append("$MEDIATOR/$mediator;") }
        }.toString()

        val deviceInfoHeader = StringBuilder().apply {
            append("$OS/Android;")
            androidOSVersion?.let { append("$OS_VERSION/${androidOSVersion};") }
        }.toString()

        append("X-Moloco-App-Info", appInfoHeader)
        append("X-Moloco-Device-Info", deviceInfoHeader)
        append("X-Moloco-SDK-Info", sdkInfoHeader)
    }
}
