package com.moloco.sdk.acm.services

import androidx.annotation.VisibleForTesting
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.ProcessLifecycleOwner
import com.moloco.sdk.acm.eventprocessing.DBWorkRequest
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.concurrent.atomic.AtomicBoolean

/**
 * Tracks the foreground and background of the application
 */
internal interface ApplicationLifecycleTracker {

    /**
     * Allows the lifecycle tracker to start observing application lifecycle events.
     */
    suspend fun startObserving()
}

/**
 * An application lifecycle tracker that tracks the bg of the application.
 *
 */
internal class ApplicationLifecycleTrackerImpl(
    private val lifecycle: Lifecycle,
    private val bgListener: ApplicationLifecycleObserver
) : ApplicationLifecycleTracker {
    companion object {
        private const val TAG = "ApplicationLifecycleTrackerServiceImpl"
    }

    private var hasRegisteredListener = AtomicBoolean(false)

    override suspend fun startObserving() = withContext(Dispatchers.Main.immediate) {
        if (hasRegisteredListener.compareAndSet(false, true)) {
            MolocoMetricsLogger.debug(TAG, "Start observing application lifecycle events")
            lifecycle.addObserver(bgListener)
        }
    }

    /**
     * Triggers an app background.
     * We will only use this specifically for instrumentation tests.
     */
    @VisibleForTesting
    internal fun triggerBackgroundEvent() {
        bgListener.onStop(ProcessLifecycleOwner.get())
    }
}

/**
 * A custom background listener that records the background event and makes a request to purge the DB.
 * This will continue observing until the application is closed.
 * Note that any backgrounding, including CTA will be considered a background event.
 */
private const val TAG = "ApplicationLifecycleObserver"

internal class ApplicationLifecycleObserver(
    private val dbWorkRequest: DBWorkRequest,
    private val scope: CoroutineScope
) : DefaultLifecycleObserver {
    override fun onStop(owner: LifecycleOwner) {
        super.onStop(owner)
        MolocoMetricsLogger.debug(TAG, "Application onStop")

        scope.launch {
            dbWorkRequest.uploadAndPurge()
        }
    }
}
