package com.moloco.sdk.acm

import androidx.annotation.VisibleForTesting
import androidx.lifecycle.ProcessLifecycleOwner
import com.moloco.sdk.acm.db.MetricsDb
import com.moloco.sdk.acm.eventprocessing.DBWorkRequestImpl
import com.moloco.sdk.acm.eventprocessing.EventProcessor
import com.moloco.sdk.acm.eventprocessing.EventProcessorImpl
import com.moloco.sdk.acm.eventprocessing.RequestSchedulerTimer
import com.moloco.sdk.acm.services.ApplicationLifecycleTrackerImpl
import com.moloco.sdk.acm.services.MolocoMetricsLogger
import com.moloco.sdk.acm.services.ApplicationLifecycleObserver
import com.moloco.sdk.acm.services.TimeProviderServiceImpl
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.launch
import java.lang.IllegalStateException
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.atomic.AtomicReference

/**
 * Callback interface for initialization of the AndroidClientMetrics Library.
 */
interface AndroidClientMetricsCallback {
    /**
     * Called when initialization of AndroidClientMetrics is successful.
     */
    fun onInitializationSuccess()

    /**
     * Called when initialization of AndroidClientMetrics fails.
     * @param e The exception that caused the initialization failure.
     */
    fun onInitializationFailure(e: Exception)
}

/**
 * Enum for giving us the initialization status of the Android Client Metrics library
 */
internal enum class InitializationStatus {
    /**
     * Library is initialized
      */
    INITIALIZED,
    /**
     * Library is initializing
     */
    INITIALIZING,
    /**
     * Library is not initialized
     */
    UNINITIALIZED
}

/**
 * Object responsible for handling analytics events for Moloco SDK.
 */
object AndroidClientMetrics {
    private lateinit var eventProcessor: EventProcessor
    private lateinit var applicationLifecycleTracker: ApplicationLifecycleTrackerImpl
    private val ioScope = CoroutineScope(Dispatchers.IO + SupervisorJob())

    private val _initializationStatus = AtomicReference(InitializationStatus.UNINITIALIZED)
    // Events stored before initialization and to be process once ACM has initialized
    private val timerList = CopyOnWriteArrayList<TimerEvent>()
    private val countList = CopyOnWriteArrayList<CountEvent>()
    private const val TAG = "AndroidClientMetrics"

    /**
     * Property that returns initialization state of the AndroidClientMetrics Library
     */
    internal val initializationStatus: InitializationStatus
        get() = _initializationStatus.get()

    /**
     * Initializes the AndroidClientMetrics with the given configuration.
     *
     * @param config The configuration used to initialize the analytics system.
     */
    fun initialize(config: InitConfig, callback: AndroidClientMetricsCallback? = null) {
        if (_initializationStatus.compareAndSet(InitializationStatus.UNINITIALIZED, InitializationStatus.INITIALIZING)) {
            ioScope.launch {
                try {
                    val metricsDAO = MetricsDb.getInstance(config.context).operationalMetricsDao()
                    val timeProviderService = TimeProviderServiceImpl()
                    val dbWorkRequest = DBWorkRequestImpl(config)

                    val requestScheduler = RequestSchedulerTimer(
                        dbWorkRequest,
                        config.requestPeriodSeconds
                    )

                    applicationLifecycleTracker = ApplicationLifecycleTrackerImpl(
                        ProcessLifecycleOwner.get().lifecycle,
                        ApplicationLifecycleObserver(dbWorkRequest, ioScope)
                    )

                    eventProcessor = EventProcessorImpl(
                        metricsDAO,
                        timeProviderService,
                        requestScheduler,
                        applicationLifecycleTracker
                    )

                    _initializationStatus.set(InitializationStatus.INITIALIZED)
                    processQueuedEvents()
                    callback?.onInitializationSuccess()
                } catch (e: IllegalStateException) {
                    MolocoMetricsLogger.error(MetricsDb.TAG, "Unable to create metrics db", e)
                    _initializationStatus.set(InitializationStatus.UNINITIALIZED)
                    callback?.onInitializationFailure(e)
                } catch (e: Exception) {
                    MolocoMetricsLogger.error(TAG, "Initialization error", e)
                    _initializationStatus.set(InitializationStatus.UNINITIALIZED)
                    callback?.onInitializationFailure(e)
                }
            }
        }
    }

    /**
     * Records a count event.
     *
     * @param event The operational count event to be recorded.
     */
    fun recordCountEvent(event: CountEvent) {
        if (_initializationStatus.get() != InitializationStatus.INITIALIZED) {
            countList.add(event)
            MolocoMetricsLogger.debug(TAG, "Moloco Client Metrics not initialized")
            return
        }
        ioScope.launch {
            eventProcessor.processCountEvent(event)
        }
    }

    /**
     * Starts a timer event for analytics tracking.
     *
     * This function initializes an `AnalyticsTimerEvent` with the provided `eventName`
     * and starts the timer for tracking the event's duration.
     * We can return an event regardless if the Library is initialized or not.
     *
     * @param eventName the name of the event to track.
     * @return an instance of `AnalyticsTimerEvent` with the timer started.
     *
     */
    fun startTimerEvent(eventName: String): TimerEvent {
        if (_initializationStatus.get() != InitializationStatus.INITIALIZED) {
            MolocoMetricsLogger.debug(TAG, "Moloco Client Metrics not initialized")
        }
        return TimerEvent.create(eventName).apply { startTimer() }
    }

    /**
     * Records a timer event.
     *
     * @param event The operational timer event to be recorded.
     */
    fun recordTimerEvent(event: TimerEvent) {
        if (_initializationStatus.get() != InitializationStatus.INITIALIZED) {
            timerList.add(event)
            MolocoMetricsLogger.debug(TAG, "Moloco Client Metrics not initialized")
            return
        }
        event.stopTimer()
        ioScope.launch {
            eventProcessor.processTimerEvent(event)
        }
    }

    /**
     * Processes events that were stored before the ACM library was initialized
     */
    private fun processQueuedEvents() {
        ioScope.launch {
            timerList.forEach { eventProcessor.processTimerEvent(it) }
            countList.forEach { eventProcessor.processCountEvent(it) }
            timerList.clear()
            countList.clear()
        }
    }

    /**
     * Triggers an app background.
     * We will only use this specifically for instrumentation tests.
     */
    @VisibleForTesting
    internal fun triggerBackgroundEvent() {
        if (this::applicationLifecycleTracker.isInitialized) {
            applicationLifecycleTracker.triggerBackgroundEvent()
        }
    }
}
