package com.moloco.sdk.acm.eventprocessing

import com.moloco.sdk.acm.db.MetricsDAO
import com.moloco.sdk.acm.http.PostMetricsRequest
import com.moloco.sdk.acm.services.MolocoMetricsLogger
import io.ktor.http.HeadersBuilder

/**
 * Interface that contains logic for sending out and purging room database
 */
interface RequestAndPurgeDB {
    suspend operator fun invoke(): Result<String>
}

/**
 * Class responsible for sending out and purging room database
 *
 * @property metricsRequest Sends the metrics request
 * @property metricsDAO DAO for DB access
 * @property dataAgeChecker Checks the age of the data before sending a metrics request
 * @property headers Headers for the metrics request
 */
internal class RequestAndPurgeDBImpl(
    private val metricsRequest: PostMetricsRequest,
    private val metricsDAO: MetricsDAO,
    private val dataAgeChecker: DataAgeChecker,
    private val headers: HeadersBuilder.() -> Unit
): RequestAndPurgeDB {
    private val TAG = "RequestAndPurgeDB"

    override suspend operator fun invoke(): Result<String> {

        // TODO: Determine delete pattern and edge cases
        // 1. What happens if app is closed during deleteAndReturnDeletedEvents
        // 2. What to do with data if metrics request fails
        // 5xx or network failures - should have http level retries
        // 4xx other errors - drop the payload and forget
        val dataToDelete = metricsDAO.deleteAndReturnDeletedEvents()
        MolocoMetricsLogger.info(TAG, "${dataToDelete.size} events processed.")

        val requestParams = DBEventToRequestTransformerImpl(dataAgeChecker)
            .transformData(dataToDelete)

        val requestResult = metricsRequest.execute(
            requestParams,
            headers
        )

        return requestResult.onSuccess {
            MolocoMetricsLogger.info(TAG, "Request Success")
        }.onFailure { exception ->
            MolocoMetricsLogger.error(TAG, "Request failure: ${exception.message}")
        }
    }
}
