package com.moloco.sdk.acm.http

import io.ktor.http.HeadersBuilder

/**
 * Helper class for building headers required for ACM requests.
 */
internal class AcmHeadersBuilder {

    internal companion object {
        // Mediator
        const val MEDIATOR = "Mediator"

        // Header key for operating system information version.
        const val OS_VERSION = "osv"

        // Header key for operating system information.
        const val OS = "OS"

        // Header key for application key.
        const val APP_KEY = "AppKey"

        // Header key for application bundle identifier.
        const val APP_BUNDLE = "AppBundle"

        // Header key for application version.
        const val APP_VERSION = "AppVersion"

        // Header key for sdk version.
        const val MOLOCO_SDK_VERSION = "SdkVersion"
    }

    /**
     * Creates headers for ACM requests based on provided parameters.
     *
     * @param sdkVersion The version of the SDK.
     * @param androidOSVersion The version of the Android operating system.
     * @param key The application key.
     * @param bundle The application bundle identifier.
     * @param appVersion The application version.
     * @return Lambda function that configures headers for ACM requests.
     */
    fun createHeaders(
        sdkVersion: String?,
        androidOSVersion: String?,
        key: String?,
        bundle: String?,
        appVersion: String?,
        mediator: String?
    ): HeadersBuilder.() -> Unit = {
        val appInfoHeader = StringBuilder().apply {
            bundle?.let { append("$APP_BUNDLE/$bundle;") }
            appVersion?.let { append("$APP_VERSION/$appVersion;") }
            key?.let { append("$APP_KEY/$key;") }
        }.toString()

        val sdkInfoHeader = StringBuilder().apply {
            sdkVersion?.let { append("$MOLOCO_SDK_VERSION/$sdkVersion;") }
            mediator?.let { append("$MEDIATOR/$mediator;") }
        }.toString()

        val deviceInfoHeader = StringBuilder().apply {
            append("$OS/Android;")
            androidOSVersion?.let { append("$OS_VERSION/${androidOSVersion};") }
        }.toString()

        append("X-Moloco-App-Info", appInfoHeader)
        append("X-Moloco-Device-Info", deviceInfoHeader)
        append("X-Moloco-SDK-Info", sdkInfoHeader)
    }
}
