package com.moloco.sdk.acm

/**
 * Class representing an analytics count event in Moloco Analytics.
 * The Count Event allows us to capture any int value.
 * It can be used to capture the number of times its recorded or record an event in history.
 *
 * @param eventName The name of the operational count event.
 */
class CountEvent(eventName: String): Event {
    companion object {
        /**
         * The maximum number of tags allowed per event.
         */
        private const val MAX_TAGS_PER_EVENT = 10

        /**
         * The maximum character length allowed for a tag.
         */
        private const val MAX_TAG_CHAR_LEN = 50
    }

    /**
     * The list of tags associated with the event.
     */
    override val eventTags = mutableListOf<EventTag>()

    /**
     * The name of the event.
     */
    override val name = eventName

    /**
     * Count value for the event. By default it is 1.
     */
    var countValue: Int = 1

    /**
     * Sets the count value for the AnalyticsCountEvent.
     *
     * @param count The count value to be set. It must be non-negative.
     * @return The modified AnalyticsCountEvent object with the count value set.
     * @throws IllegalArgumentException If the count provided is negative.
     */
    fun withCount(count: Int): CountEvent {
        require(count >= 0) { "Count cannot be negative" }
        return this.apply { countValue = count }
    }

    /**
     * Adds a tag to the operational count event.
     *
     * @param key The key of the tag.
     * @param value The value of the tag.
     * @return The updated OperationalCountEvent with the new tag.
     */
    override fun withTag(key: String, value: String): CountEvent {
        if (eventTags.size >= MAX_TAGS_PER_EVENT
            || key.length > MAX_TAG_CHAR_LEN
            || value.length > MAX_TAG_CHAR_LEN
        ) {

            if (BuildConfig.DEBUG) {
                throw IllegalArgumentException("Exceeded tag constraints: MAX_TAGS_PER_EVENT = $MAX_TAGS_PER_EVENT, MAX_TAG_CHAR_LEN = $MAX_TAG_CHAR_LEN")
            }

            return this
        }
        eventTags.add(EventTag(key, value))
        return this
    }
}
