package com.moloco.sdk.acm.eventprocessing

import com.moloco.sdk.MetricsRequest
import com.moloco.sdk.acm.db.EventEntity
import com.moloco.sdk.acm.db.EventType
import com.moloco.sdk.acm.http.MetricsRequestParams

/**
 * Interface for transforming a list of database event entities into request parameters
 * for metrics reporting.
 *
 * @constructor Creates a new instance of the transformer. This is intended for internal use only.
 */
internal interface DBEventToRequestTransformer {
    /**
     * Transforms a list of event entities along with client-specific options into
     * a set of parameters suitable for metrics requests.
     *
     * @param events A list of `EventEntity` objects representing the events to be transformed.
     *               Each event contains information about a specific occurrence
     *               that needs to be processed for metrics.
     * @return A `MetricsRequestParams` object that contains the transformed
     *         parameters ready to be used in a metrics request.
     */
    fun transformData(events: List<EventEntity>) : MetricsRequestParams
}

internal class DBEventToRequestTransformerImpl(private val dataAgeChecker: DataAgeChecker): DBEventToRequestTransformer {

    override fun transformData(events: List<EventEntity>) : MetricsRequestParams {
        val timers: MutableList<MetricsRequest.PostMetricsRequest.TimerEvent> = mutableListOf()
        val counters: MutableList<MetricsRequest.PostMetricsRequest.CountEvent> = mutableListOf()

        for (event in events) {
            if (!dataAgeChecker.isEventOutdated(event)) {
                when(event.eventType) {
                    EventType.COUNT -> {
                        val metric = MetricsRequest.PostMetricsRequest.CountEvent.newBuilder()
                            .setName(event.name)
                            .addAllTags(event.tags)
                        event.data?.let {metric.setCount(it.toInt()) }
                        counters.add(metric.build())
                    }

                    EventType.TIMER -> {
                        val metric = MetricsRequest.PostMetricsRequest.TimerEvent.newBuilder()
                            .setName(event.name)
                            .addAllTags(event.tags)
                        event.data?.let { metric.setElapsedTimeMillis(it) }
                        timers.add(metric.build())
                    }
                }
            }
        }

        return MetricsRequestParams(
            counters,
            timers
        )
    }
}
