package com.moloco.sdk.acm.eventprocessing

import com.moloco.sdk.acm.eventprocessing.DefaultDataTimeConfigurations.REQUEST_POLLING_INTERVAL_SECONDS
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.concurrent.Executors
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicBoolean

/**
 * Interface representing a scheduler for managing data upload and purge operations.
 */
internal interface RequestScheduler {
    /**
     * Schedules an upload of data and purges old data.
     * This method is responsible for initiating the upload process and
     * ensuring that outdated data is removed.
     */
    fun scheduleUploadAndPurge()
}

internal class RequestSchedulerTimer(
    private val dbWorkRequest: DBWorkRequest,
    private val requestPeriodSeconds: Long = REQUEST_POLLING_INTERVAL_SECONDS, // 10 minutes by default; This will be configurable
    private val scheduler: ScheduledExecutorService = Executors.newSingleThreadScheduledExecutor(),
    private val coroutineScope: CoroutineScope
): RequestScheduler {

    private val isRunning = AtomicBoolean(false)

    override fun scheduleUploadAndPurge() {
        if (isRunning.compareAndSet(false, true)) {
            scheduler.scheduleWithFixedDelay({
                coroutineScope.launch {
                    dbWorkRequest.uploadAndPurge()
                }
            }, requestPeriodSeconds, requestPeriodSeconds, TimeUnit.SECONDS)
        }
    }
}
