package com.moloco.sdk.acm.http

import com.moloco.sdk.MetricsRequest
import com.moloco.sdk.acm.services.MolocoMetricsLogger
import io.ktor.client.HttpClient
import io.ktor.client.plugins.timeout
import io.ktor.client.request.headers
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.client.statement.HttpResponse
import io.ktor.client.statement.bodyAsText
import io.ktor.http.ContentType
import io.ktor.http.HeadersBuilder
import io.ktor.http.HttpStatusCode
import io.ktor.http.contentType

/**
 * Interface for making asynchronous POST requests for metrics data.
 */
internal interface PostMetricsRequest {
    /**
     * Executes a POST request with metrics request parameters and custom headers.
     *
     * @param metricsRequestParams Parameters for metrics request.
     * @param headers Lambda function to configure HTTP headers.
     * @return Response body as a String, or null if request fails.
     */
    suspend fun execute(metricsRequestParams: MetricsRequestParams,
                        headers: HeadersBuilder.() -> Unit) : Result<String>
}

/**
 * Implementation of PostMetricsRequest interface using HttpClient for making POST requests.
 *
 * @property httpClient HttpClient instance for making HTTP requests.
 * @property apiUrl URL endpoint for the metrics API.
 */
internal class PostMetricsRequestImpl(
    private val httpClient: HttpClient,
    private val apiUrl: String,
) : PostMetricsRequest {
    private val TAG = "PostMetricsRequest"

    /**
     * Executes a POST request with metrics request parameters and custom headers.
     *
     * @param metricsRequestParams Parameters for metrics request.
     * @param headers Lambda function to configure HTTP headers.
     * @return Response body as a String, or null if request fails.
     */
    override suspend fun execute(
        metricsRequestParams: MetricsRequestParams,
        headers: HeadersBuilder.() -> Unit
    ) : Result<String> = makeMetricsRequest(
        timeoutMillis = 5000L,
        metricsRequestParams = metricsRequestParams,
        headers = headers
    )

    /**
     * Makes a POST request with configured timeout, metrics request parameters, and custom headers.
     *
     * @param timeoutMillis Timeout value in milliseconds for the HTTP request.
     * @param metricsRequestParams Parameters for metrics request.
     * @param headers Lambda function to configure HTTP headers.
     * @return Response body as a String, or null if request fails.
     */
    private suspend fun makeMetricsRequest(
        timeoutMillis: Long,
        metricsRequestParams: MetricsRequestParams,
        headers: HeadersBuilder.() -> Unit
    ): Result<String> {
        return try {
            val response: HttpResponse = httpClient.post(apiUrl) {
                timeout { requestTimeoutMillis = timeoutMillis }
                contentType(ContentType.Application.ProtoBuf)
                headers(headers)
                setBody(createHttpClientBody(metricsRequestParams))
            }

            val responseStatus = response.status
            val responseBody = response.bodyAsText()

            if (responseStatus == HttpStatusCode.OK) {
                MolocoMetricsLogger.info(TAG, "Post Metrics Request Success: $responseBody")
                Result.success(responseBody)
            } else {
                MolocoMetricsLogger.error(TAG, "Post Metrics Request Error: $responseBody")
                Result.failure(Exception("PostMetricsRequest Error: $responseStatus"))
            }
        } catch (e: Exception) {
            MolocoMetricsLogger.error(TAG, "Post Metrics Request Exception", e)
            Result.failure(e)
        }
    }

    /**
     * Converts metrics request parameters to a byte array suitable for HttpClient body.
     *
     * @param params Parameters for metrics request.
     * @return Byte array representation of metrics request.
     */
    private fun createHttpClientBody(params: MetricsRequestParams): ByteArray {
        val metricsRequest: MetricsRequest.PostMetricsRequest = MetricsRequest.PostMetricsRequest.newBuilder()
            .addAllCounts(params.counts)
            .addAllDurations(params.timers)
            .build()
        return metricsRequest.toByteArray()
    }
}
