package com.moloco.sdk.acm.services

import android.text.TextUtils
import android.util.Log
import com.moloco.sdk.acm.BuildConfig
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * Singleton object responsible for logging within the Moloco SDK.
 *
 * MolocoLogger provides a centralized logging mechanism with the ability to enable or disable
 * logging globally.
 *
 * This is a copy of the Moloco Logger in the moloco-sdk module.
 * We will move this to a separate module: https://mlc.atlassian.net/browse/SDK-2363
 *
 * @property logEnabled Indicates whether logging is globally enabled.
 */
internal object MolocoMetricsLogger {
    private val scope = CoroutineScope(Dispatchers.Main)

    /**
     * List of LoggerListener instances subscribed to receive log events.
     */
    private val listeners = arrayListOf<LoggerListener>()

    /**
     * Indicates whether logging is globally enabled.
     * In release builds, all operations are halted unless explicitly enabled.
     * By default, logging is enabled in debug builds or when the
     * "debug.moloco.enable_logs" ADB property is set to true.
     */
    @JvmStatic
    var logEnabled: Boolean = BuildConfig.DEBUG || AdbUtil.canShowLogs("debug.moloco.enable_logs")


    // TODO. Tags depending on caller class.
    private const val MOLOCO_TAG = "ACM"

    /**
     * Used for the Demo app to display logs in its event logger
     */
    @JvmStatic
    fun addLoggerListener(loggerListener: LoggerListener) {
        listeners.add(loggerListener)
    }

    /**
     * Logs a message with the specified tag and message.
     *
     * @param tag The tag to associate with the log message. Default is [MOLOCO_TAG].
     * @param msg The message to log.
     * @param forceLogging If set to true, the message will be logged even if logging is not enabled. Default is false.
     */
    fun debug(tag: String = MOLOCO_TAG, msg: String, forceLogging: Boolean = false) {
        if (!logEnabled && !forceLogging) return

        val formattedTag = tag.prefixWithMolocoName()
        val formattedMsg = msg.prefixWithMethodName()

        Log.d(formattedTag, formattedMsg)
        fireListeners(formattedTag, formattedMsg)
    }

    /**
     * Logs a message with the specified tag and message.
     *
     * @param tag The tag to associate with the log message. Default is [MOLOCO_TAG].
     * @param msg The message to log.
     * @param forceLogging If set to true, the message will be logged even if logging is not enabled. Default is false.
     */
    fun info(tag: String = MOLOCO_TAG, msg: String, forceLogging: Boolean = false) {
        if (!logEnabled && !forceLogging) return

        val formattedTag = tag.prefixWithMolocoName()
        val formattedMsg = msg.prefixWithMethodName()

        Log.i(formattedTag, formattedMsg)
        fireListeners(formattedTag, formattedMsg)
    }

    /**
     * Logs a message with the specified tag and message.
     *
     * @param tag The tag to associate with the log message. Default is [MOLOCO_TAG].
     * @param msg The message to log.
     * @param forceLogging If set to true, the message will be logged even if logging is not enabled. Default is false.
     */
    fun warn(tag: String = MOLOCO_TAG, msg: String, exception: Throwable? = null, forceLogging: Boolean = false) {
        if (!logEnabled && !forceLogging) return

        val formattedTag = tag.prefixWithMolocoName()
        val formattedMsg = msg.prefixWithMethodName()

        Log.w(formattedTag, formattedMsg, exception)
        fireListeners(formattedTag, formattedMsg)
    }

    /**
     * Logs a message with the specified tag and message.
     *
     * @param tag The tag to associate with the log message. Default is [MOLOCO_TAG].
     * @param msg The message to log.
     * @param forceLogging If set to true, the message will be logged even if logging is not enabled. Default is false.
     */
    fun error(
        tag: String = MOLOCO_TAG,
        msg: String,
        exception: Throwable? = null,
        forceLogging: Boolean = false
    ) {
        if (!logEnabled && !forceLogging) return

        val formattedTag = tag.prefixWithMolocoName()
        val formattedMsg = msg.prefixWithMethodName()

        Log.e(formattedTag, formattedMsg, exception)
        fireListeners(formattedTag, formattedMsg)
    }

    /**
     * For internal purposes only. Useful for quick logs. The default tag is `==tlog==`
     * and the [msg] will be prefixed with method name.
     */
    fun tlog(msg: String) {
        val formattedMsg = msg.prefixWithMethodName()

        Log.i("==tlog==", formattedMsg)
    }

    @Suppress("UNUSED_VARIABLE")
    fun getCallingMethodName(): String {
        val stackTraceArray: Array<StackTraceElement> = Throwable().stackTrace

        val stackTraceElement = findMostRelevantStackTrace(stackTraceArray)
        val className = stackTraceElement.className
        val methodName = stackTraceElement.methodName

        val clazz = Class.forName(className)
        val clazzCanonical = clazz.canonicalName
        val clazzAnonymous = clazz.isAnonymousClass
        val clazzMethods = clazz.declaredMethods
        return stackTraceElement.methodName.let {
            if (it == "invokeSuspend") {
                // TODO: check this page for possible better implementation
                //    https://github.com/JakeWharton/timber/blob/trunk/timber/src/main/java/timber/log/Timber.kt

                /**
                 * From this string:
                 * foo.bar.RewardedMock$startRewardedVideoLogic$1
                 * creates this one:
                 * startRewardedVideoLogic
                 */
                val substring = stackTraceElement.className.removeSuffix(
                    "$1"
                ).substringAfterLast("$")

                substring
            } else {
                it
            }
        }
    }

    private fun fireListeners(tag: String, msg: String) {
        /**
         * Sometimes logs can come from non ui thread
         */
        scope.launch {
            listeners.forEach {
                it.onLog(tag.prefixWithMolocoName(), msg)
            }
        }
    }

    /**
     * Prefixes [this] with [MOLOCO_TAG]. If already exists, then returns [this]
     */
    private fun String.prefixWithMolocoName() = if (startsWith(
            MOLOCO_TAG
        )
    ) {
        this
    } else {
        MOLOCO_TAG + this
    }

    /**
     * Tries to attach caller method name to [this]. If unsuccessful returns [this]
     */
    private fun String.prefixWithMethodName() = try {
        "[${getCallingMethodName()}] $this"
    } catch (e: Exception) {
        this
    }

    /**
     * Returns the most relevant stack trace element by skipping over elements
     * belonging to the MolocoLogger class.
     *
     * @param stackTraceArray The array of stack trace elements to search through.
     * @return The most relevant stack trace element.
     */
    private fun findMostRelevantStackTrace(
        stackTraceArray: Array<StackTraceElement>
    ): StackTraceElement {
        stackTraceArray.forEach { stackTraceElement ->
            if (stackTraceElement.className == MolocoMetricsLogger.javaClass.canonicalName) {
                // omit it
            } else {
                return@findMostRelevantStackTrace stackTraceElement
            }
        }

        // if no relevant element is found, return the first one
        return stackTraceArray.first()
    }

    interface LoggerListener {
        fun onLog(tag: String, msg: String)
    }
}

private object AdbUtil {
    /**
     * Checks if we can show logs. This param can be set like this in command shell:
     * "adb shell setprop debug.moloco.enable_logs true"
     *
     * @return true if the 'debug.moloco.enable_logs=true' property is set.
     */
    fun canShowLogs(key: String): Boolean {
        val enableLogsProperty: String? = getSystemProperty(key)
        return enableLogsProperty.toBoolean()
    }

    /**
     * By using reflection we can access the system properties which otherwise could only be be had by running this command in command shell:
     * "adb shell getprop"
     *
     * @param key the name of the property
     * @return value of the property or null
     */
    private fun getSystemProperty(key: String): String? {
        var value: String? = null
        try {
            value = Class.forName("android.os.SystemProperties")
                .getMethod("get", String::class.java).invoke(null, key) as String
            if (TextUtils.isEmpty(value)) {
                value = null
            }
        } catch (e: Exception) {
            // Something went wrong or we were only running unit test
            //  e.printStackTrace()
        }
        return value
    }
}