package com.moloco.sdk.internal

import android.net.Uri
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.utils.substituteErrorMacro
import com.moloco.sdk.internal.utils.substituteTimestampMacro
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MolocoAdSubErrorType
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest

internal interface SdkEventUrlTracker {
    operator fun invoke(
        url: String,
        timestamp: Long,
        error: MolocoInternalAdError? = null,
    ): Boolean
}

internal fun SdkEventUrlTracker(): SdkEventUrlTracker = Instance

private val Instance by lazy {
    SdkEventUrlTrackerImpl(
        SdkObjectFactory.Network.persistentHttpRequestSingleton
    )
}

@VisibleForTesting
internal class SdkEventUrlTrackerImpl(
    private val persistentHttpRequest: PersistentHttpRequest,
) : SdkEventUrlTracker {
    override fun invoke(url: String, timestamp: Long, error: MolocoInternalAdError?): Boolean = try {
        val preparedUrl = Uri.parse(url.substituteMacros(timestamp, error?.subErrorType)).buildUpon()
            .build()

        persistentHttpRequest.send(preparedUrl.toString())
        true
    } catch (e: Exception) {
        MolocoLogger.error("SdkEventUrlTrackerImpl", e.toString())
        false
    }

    private fun String.substituteMacros(timestamp: Long, errorSubtype: MolocoAdSubErrorType?): String {
        var substitutedUrl = errorSubtype?.let {substituteErrorMacro(errorSubtype.metricsRepresentation)} ?: this
        substitutedUrl = substitutedUrl.substituteTimestampMacro(timestamp)
        return substitutedUrl
    }
}

private const val ERROR_CODE_MACRO = "[ERROR_CODE]"
private const val TIMESTAMP_MACRO = "[HAPPENED_AT_TS]"