package com.moloco.sdk.internal.ilrd

import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.services.TimeProviderService
import kotlinx.serialization.Serializable
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json
import java.util.UUID
import java.util.concurrent.atomic.AtomicReference

/**
 * Represents an active ILRD (Impression Level Revenue Data) session.
 * Tracks session information, timestamps, and impression counts for various ad formats.
 *
 * This class supports serialization to JSON for persistence across app lifecycle events or app restarts.
 * When provided with a serialized JSON string in the constructor, it will attempt to restore its state from that JSON.
 */
internal class IlrdActiveSession(
    private val timeProvider: TimeProviderService,
    /**
     * Optional JSON string from a previously serialized session.
     * If provided, the session will restore its state from this string.
     * If null or invalid, a new session will be created with default values.
     */
    recoveredSessionJson: String? = null,
) {
    /** Unique identifier for this session */
    val sessionId by lazy { restoredSessionData?.sessionId ?: UUID.randomUUID().toString() }

    /**
     * The current impression counts for this session, across different ad formats.
     * Updated whenever new impressions are received.
     */
    val impressionCounts: ImpressionCounts get() = _impressionCounts.get()

    /**
     * Timestamp when this session was created, in milliseconds since epoch.
     * For restored sessions, this is the original creation time, not the restoration time.
     */
    val sessionStartTs: Long by lazy { restoredSessionData?.sessionStartTs ?: timeProvider.currentTime() }

    /**
     * Flag indicating whether this session has expired.
     * A session expires based on inactivity or maximum length rules.
     */
    val isExpired get() = _isExpired

    /**
     * Deserialized session data from the JSON string provided in the constructor.
     * This property is lazily initialized to handle deserialization errors gracefully.
     * If the JSON is invalid, this will be null and the session will initialize with default values.
     *
     * The parsed data is used to initialize the session's properties:
     * - sessionId
     * - impressionCounts
     * - expired state
     * - sessionStartTs
     */
    private val restoredSessionData by lazy {
        recoveredSessionJson?.let {
            try {
                Json.decodeFromString<SessionData>(it)
            } catch (e: Exception) {
                MolocoLogger.error(TAG, "Error deserializing session data", e)
                null
            }
        }
    }

    private val _impressionCounts = AtomicReference(
        restoredSessionData?.impressionCounts ?: ImpressionCounts(-1, 0, 0, 0, 0, 0)
    )

    private var _isExpired: Boolean = restoredSessionData?.isExpired ?: false

    /**
     * Updates the appropriate impression counter in the session based on ad format.
     *
     * Note: This method is called only under mutex protection in [IlrdEventsRepository],
     * so no concurrent updates are possible. AtomicReference is used only for thread-safe reads.
     *
     * @param ilrdData The impression data to process
     */
    fun updateImpressionCount(ilrdData: IlrdProvider.IlrdImpression) {
        when (ilrdData) {
            is IlrdProvider.IlrdImpression.Max -> {
                val adFormat = ilrdData.impression.adFormat.uppercase()
                updateImpressionCountPerAdFormat(adFormat, "Applovin")
            }

            is IlrdProvider.IlrdImpression.LevelPlay -> {
                val adFormat = ilrdData.impression.adFormat.uppercase()
                updateImpressionCountPerAdFormat(adFormat, "Ironsource")
            }
        }
    }

    /**
     * Updates the impression count for a specific ad format.
     *
     * @param adFormat The ad format string (e.g., "BANNER", "MREC", "INTERSTITIAL")
     * @param mediationName The name of the mediation platform (for logging)
     * @return true if the update was successful, false if the ad format was not recognized
     */
    private fun updateImpressionCountPerAdFormat(adFormat: String, mediationName: String): Boolean {
        val currentTime = timeProvider.currentTime()
        val currentImpressionCount = _impressionCounts.get()

        val updatedImpressionCount = when {
            adFormat.contains("BANNER") -> {
                currentImpressionCount.copy(
                    lastEventReceivedTs = currentTime,
                    banner = currentImpressionCount.banner + 1
                )
            }

            adFormat.contains("MREC") -> {
                currentImpressionCount.copy(
                    lastEventReceivedTs = currentTime,
                    mrec = currentImpressionCount.mrec + 1
                )
            }

            adFormat.contains("NATIVE") -> {
                currentImpressionCount.copy(
                    lastEventReceivedTs = currentTime,
                    native = currentImpressionCount.native + 1
                )
            }

            adFormat.contains("INTER") -> {
                currentImpressionCount.copy(
                    lastEventReceivedTs = currentTime,
                    interstitial = currentImpressionCount.interstitial + 1
                )
            }

            adFormat.contains("REWARD") -> {
                currentImpressionCount.copy(
                    lastEventReceivedTs = currentTime,
                    rewarded = currentImpressionCount.rewarded + 1
                )
            }

            else -> {
                MolocoLogger.warn(TAG, "Unknown ad format for $mediationName: $adFormat")
                return false
            }
        }

        // Simple atomic set - no CAS loop needed since updates are mutex-protected
        _impressionCounts.set(updatedImpressionCount)
        return true
    }

    /**
     * Marks this session as expired.
     * An expired session will be replaced with a new one on the next event.
     */
    fun expire() {
        _isExpired = true
    }

    /**
     * Serializes this session to a JSON string.
     * The resulting string can be used to restore the session state later
     * by passing it to the constructor.
     *
     * @return JSON string representation of this session
     */
    fun toJson(): String {
        val data = SessionData(
            sessionId = sessionId,
            impressionCounts = impressionCounts,
            isExpired = isExpired,
            sessionStartTs = sessionStartTs
        )
        return Json.encodeToString(SessionData.serializer(), data)
    }

    /**
     * Returns a string representation of this session.
     */
    override fun toString(): String {
        val counts = impressionCounts
        val totalImpressions = counts.banner + counts.mrec + counts.native +
            counts.interstitial + counts.rewarded

        return "IlrdActiveSession(id=$sessionId, " +
            "startTs=$sessionStartTs, " +
            "expired=$isExpired, " +
            "impressions=$totalImpressions [banner=${counts.banner}, " +
            "mrec=${counts.mrec}, " +
            "native=${counts.native}, " +
            "interstitial=${counts.interstitial}, " +
            "rewarded=${counts.rewarded}])"
    }

    /**
     * Private data class used for serialization/deserialization.
     * Contains all the session state that needs to be persisted.
     */
    @Serializable
    private data class SessionData(
        val sessionId: String,
        val impressionCounts: ImpressionCounts,
        val isExpired: Boolean,
        val sessionStartTs: Long,
    )

    /**
     * Data class representing counters for different types of ad impressions.
     */
    @Serializable
    data class ImpressionCounts(
        val lastEventReceivedTs: Long,
        val banner: Int,
        val mrec: Int,
        val native: Int,
        val interstitial: Int,
        val rewarded: Int,
    )

    companion object {
        private const val TAG = "IlrdActiveSession"
    }
}
