package com.moloco.sdk.internal.ilrd

import androidx.annotation.VisibleForTesting
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import com.moloco.sdk.IlrdRequest
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.internal.services.AdData
import com.moloco.sdk.internal.services.AdDataService
import com.moloco.sdk.internal.services.DataStoreService
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.internal.utils.launchUnit
import com.moloco.sdk.xenoss.sdkdevkit.android.persistenttransport.PersistentHttpRequest
import io.ktor.http.ContentType
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.coroutines.withContext
import java.io.ByteArrayOutputStream
import java.util.UUID
import java.util.zip.GZIPOutputStream
import kotlin.time.Duration

/**
 * Repository for collecting, batching, and sending Impression Level Revenue Data (ILRD) events.
 *
 * This class is responsible for both event collection and session management, as these concerns
 * are tightly coupled - events affect session metrics, and the session state determines when
 * events should be sent.
 *
 * The repository uses several strategies to determine when to send batched events:
 * 1. When maximum batch size is reached ([maxBatchSize])
 * 2. When the upload interval has elapsed since last send ([uploadInterval])
 * 3. When the session has expired based on inactivity ([sessionExp])
 * 4. When the app goes to background (via [Lifecycle] observation)
 *
 * All operations are thread-safe, protected by a mutex to ensure data consistency
 * when called from multiple threads or coroutines.
 *
 * Session handling:
 * - Session state is managed internally to ensure thread-safety and consistency
 * - Session is created and updated atomically with event processing
 * - External components access session data via immutable snapshots only
 *
 * For error handling and network reliability, this class uses [PersistentHttpRequest]
 * which provides automatic retries and persistence of requests during network failures.
 */
internal class IlrdEventsRepository(
    private val scope: CoroutineScope,
    private val url: String,
    private val persistentHttpRequest: PersistentHttpRequest,
    /**
     * Duration after last impression to start a new session
     */
    private val sessionExp: Duration,
    /**
     * How many events to set in a batch. Takes priority over upload duration
     */
    private val maxBatchSize: Int,
    /**
     * If uploadInterval is still not reached but the [maxBatchSize] is already at max count,
     * then we should send it, irrespective of uploadInterval
     */
    private val uploadInterval: Duration,
    private val sessionMaxLength: Duration,
    private val timeProvider: TimeProviderService,
    processLifeycle: Lifecycle,
    private val advertisingIdService: AdDataService,
    private val pubId: String,
    private val appId: String,
    private val dataStoreService: DataStoreService,
    /**
     * Scheduler for session expiry based on inactivity.
     */
    @get:VisibleForTesting
    val sessionExpiryScheduler: IlrdScheduler = IlrdScheduler(scope, timeProvider, "SessionExpiryScheduler"),
    /**
     * Scheduler for maximum session length.
     */
    @get:VisibleForTesting
    val sessionMaxLengthScheduler: IlrdScheduler = IlrdScheduler(scope, timeProvider, "SessionMaxLengthScheduler"),
    /**
     * Scheduler for scheduled uploads.
     */
    @get:VisibleForTesting
    val scheduledUploadScheduler: IlrdScheduler = IlrdScheduler(scope, timeProvider, "UploadIntervalScheduler"),
) : DefaultLifecycleObserver {
    private val mutex = Mutex()

    /**
     * The current active session.
     * This property is protected by the repository's mutex to ensure thread-safety.
     * All session updates happen atomically with event processing to maintain consistency.
     * External components should use immutable snapshots of this session instead of direct access.
     */
    var session: IlrdActiveSession? = null
        private set

    /**
     * Collected impression events waiting to be sent.
     * This list is protected by the repository's mutex for thread safety.
     */
    @VisibleForTesting
    val events = mutableListOf<IlrdRequest.ImpressionLevelRevenue>()

    init {
        MolocoLogger.info(TAG, "ILRD repository initialized - url=$url, uploadInterval=$uploadInterval, maxBatchSize=$maxBatchSize, sessionExpiry=$sessionExp, maxSessionLength=$sessionMaxLength")
        scope.launch {
            mutex.withLock {
                restoreSavedSession();
            }
            withContext(DispatcherProvider().main) {
                processLifeycle.addObserver(this@IlrdEventsRepository)
            }
        }
    }

    /**
     * Attempts to restore a previously saved session from persistent storage.
     *
     * This method is called during repository initialization to recover session state
     * after app restarts or process death. The process:
     *
     * 1. Retrieves the serialized session JSON from [DataStoreService]
     * 2. Removes the key from storage to prevent reusing stale data
     * 3. Creates a new [IlrdActiveSession] with the recovered JSON
     *
     * If no saved session exists, this method does nothing and a new session
     * will be created when needed.
     */
    private suspend fun restoreSavedSession() {
        val existingSession = dataStoreService.getString(KEY_ILRD_SESSION_STORE) ?: return
        dataStoreService.removeKey(KEY_ILRD_SESSION_STORE)

        // Create a temporary session to check validity
        val restoredSession = IlrdActiveSession(timeProvider, existingSession)

        // Check time constraints
        val currentTime = timeProvider.currentTime()
        val sessionDuration = currentTime - restoredSession.sessionStartTs
        val lastEventTime = restoredSession.impressionCounts.lastEventReceivedTs

        // Don't restore if session exceeds maximum length
        if (sessionDuration > sessionMaxLength.inWholeMilliseconds) {
            MolocoLogger.info(TAG, "Discarding restored session - exceeded maximum length")
            return
        }

        // Don't restore if inactivity timeout exceeded (if there were previous impressions)
        if (lastEventTime > 0) {
            val inactivityTime = currentTime - lastEventTime
            if (inactivityTime > sessionExp.inWholeMilliseconds) {
                MolocoLogger.info(TAG, "Discarding restored session - exceeded inactivity timeout")
                return
            }
        }

        // Session is valid, keep it
        MolocoLogger.info(TAG, "ILRD session restored successfully")
        session = restoredSession
    }

    /**
     * Notifies the repository of a bid token request.
     * This may trigger a new session if the current one has expired.
     *
     * Session validation is done in the context of bid token requests
     * to ensure proper session tracking when ads are requested.
     */
    @Synchronized
    fun onBidTokenRequest() {
        ensureValidSession()
        scheduleSessionExpiry()
    }

    /**
     * Processes an impression event with revenue data.
     *
     * This method follows a specific order of operations:
     * 1. First check if existing events need to be sent (due to time constraints)
     * 2. Then ensure we have a valid session for the new event
     * 3. Add the new event to the batch
     * 4. Finally check if batch is full and needs immediate sending
     *
     * This sequence ensures old events are sent before new ones,
     * and prevents session expiry from mixing events across sessions.
     *
     * @param ilrdData The impression event data from a mediation platform
     */
    fun onEvent(ilrdData: IlrdProvider.IlrdImpression) = scope.launchUnit {
        mutex.withLock {
            ensureValidSession()
            scheduleSessionExpiry()

            // Update session impression counts based on ad format
            session?.updateImpressionCount(ilrdData)

            // Process and add the new event to the batch
            val model = createModel(ilrdData)
            events.add(model)
            MolocoLogger.info(TAG, "Event id ${model.eventId} added. Count: ${events.size}")

            // Step 6: Check if batch is full and needs immediate sending
            // Batch size takes priority over upload interval
            if (isBatchSizeReached()) {
                sendEvents()
            }
        }
    }

    /**
     * Creates a protobuf model for the impression event,
     * linking it to the current session ID.
     *
     * @param ilrdData The impression data to convert to a protobuf model
     * @return A built ImpressionLevelRevenue protobuf message
     */
    private fun createModel(ilrdData: IlrdProvider.IlrdImpression) = IlrdRequest.ImpressionLevelRevenue.newBuilder()
        .setEventId(UUID.randomUUID().toString())
        .apply {
            session?.let { setSessionId(it.sessionId) }
                ?: MolocoLogger.warn(TAG, "createModel() Session is null")
            when (ilrdData) {
                is IlrdProvider.IlrdImpression.Max -> setMax(ilrdData.impression)
                is IlrdProvider.IlrdImpression.LevelPlay -> setLevelplay(ilrdData.impression)
            }
        }.build()

    /**
     * Lifecycle callback when app goes to background.
     * Ensure all pending events are sent before app becomes inactive.
     *
     * This method:
     * 1. Stores the current session to persistent storage for recovery
     * 2. Sends any pending events to ensure data is not lost
     *
     * @param owner The LifecycleOwner that triggered this callback
     */
    override fun onStop(owner: LifecycleOwner) = scope.launchUnit {
        mutex.withLock {
            MolocoLogger.info(TAG, "onStop called, storing session and sending events")
            session?.let {
                val serializedCurrentSession = it.toJson()
                dataStoreService.set(KEY_ILRD_SESSION_STORE, serializedCurrentSession)
            }
            sendEvents()
        }
    }

    /**
     * Creates a new session and schedules its maximum length timer.
     *
     * This method:
     * 1. Instantiates a new IlrdActiveSession
     * 2. Schedules a maximum session length timer
     * 3. Schedules the upload interval timer
     * 4. Logs the new session creation
     */
    @VisibleForTesting
    internal fun startSession() {
        session = IlrdActiveSession(timeProvider)

        // Schedule maximum session time, regardless of whether session is expired
        scheduleMaxSessionLength()

        // Schedule upload interval
        scheduleUploadIntervalScheduler()

        MolocoLogger.info(TAG, "New session started: ${session?.sessionId.toString()}")
    }

    /**
     * Ensures a valid session exists, creating a new one if necessary.
     *
     * This method checks:
     * 1. If session is null, create a new one
     * 2. If session is expired, create a new one
     *
     * Used by both [onBidTokenRequest] and [onEvent] to maintain consistent session state.
     */
    @VisibleForTesting
    internal fun ensureValidSession() {
        if (session == null || session?.isExpired == true) {
            startSession()
        }
    }

    /**
     * Schedules the session expiry task based on the configured session expiry duration.
     *
     * When the session expiry timer triggers, it:
     * 1. Marks the current session as expired
     * 2. Sends any pending events
     *
     * This ensures sessions have a limited lifetime of inactivity.
     */
    @VisibleForTesting
    internal fun scheduleSessionExpiry() {
        sessionExpiryScheduler.schedule(sessionExp) {
            session?.expire()
            sendEvents()
        }
    }

    /**
     * Schedules a task to expire the session after the maximum session length.
     *
     * This enforces a maximum duration for a session, regardless of activity.
     * When the maximum duration is reached:
     * 1. The current session is marked as expired
     * 2. Any pending events are sent immediately
     *
     * A new session will be created on the next event or bid token request.
     * This prevents sessions from remaining active indefinitely and ensures
     * reasonable segmentation of user activity.
     */
    private fun scheduleMaxSessionLength() {
        sessionMaxLengthScheduler.schedule(sessionMaxLength) {
            session?.expire()
            sendEvents()
        }
    }

    /**
     * Schedules periodic event uploads based on the configured upload interval.
     *
     * This ensures events are sent at regular intervals even if the batch size
     * threshold isn't reached. Currently disabled with an early return (true condition).
     *
     * When enabled, this would schedule regular uploads based on [uploadInterval],
     * ensuring timely delivery of impression data even during periods of low activity.
     */
    private fun scheduleUploadIntervalScheduler() {
        scheduledUploadScheduler.schedule(uploadInterval) {
            sendEvents()
        }
    }

    /**
     * Sends collected events to the server via persistent HTTP request.
     * The PersistentHttpRequest handles errors and retries automatically.
     *
     * Also updates the session's last events sent timestamp to maintain
     * proper timing for the next batch of events.
     */
    @VisibleForTesting
    internal fun sendEvents() {
        // A `sendEvents` was triggered, either by session expiry, batch size or upload interval.
        // Regardless of the method schedule a new upload interval
        scheduleUploadIntervalScheduler()

        // If no events are batched, then do nothing.
        // No point in wasting precious resources.
        if (events.isEmpty()) {
            MolocoLogger.info(TAG, "Request for sendEvent came, but event list is empty. Returning")
            return
        }

        val ilrdRequestProto = IlrdRequest.ImpressionRevenueRequest.newBuilder()
            .apply {
                setOs("Android")
                // setIdfv("TODO") // TODO tomi AppSetId SDK-4216
                setPublisherId(pubId)
                setPublisherAppId(appId)
                (advertisingIdService.advertisingData() as? AdData.Available)?.let {
                    setDeviceId(it.id)
                }

                addAllEvents(events)
            }
            .build()

        MolocoLogger.info(TAG, "Ilrd request created now sending it with ${ilrdRequestProto.eventsList.size} events")


        /**
         * Important: Must be gzipped, otherwise only about 10 events can be sent. This way the [maxBatchSize] can be 30
         */
        val body = ilrdRequestProto.toByteArray().gzipCompress()
        persistentHttpRequest.sendPost(url, body, ContentType.Application.ProtoBuf, "gzip")

        events.clear()
    }

    /**
     * Checks if the batch has reached maximum size.
     * Batch size takes priority over upload interval.
     *
     * @return true if the batch size has reached or exceeded the maximum, false otherwise
     */
    @VisibleForTesting
    fun isBatchSizeReached() = (events.size >= maxBatchSize)
        .also { if (it) MolocoLogger.info(TAG, "batch size reached") }

    companion object {
        private const val TAG = "IlrdEventsRepository"
        @VisibleForTesting
        const val KEY_ILRD_SESSION_STORE = "ilrd_session_store"

        /**
         * Compresses a byte array using GZIP compression.
         *
         * @return The GZIP-compressed byte array
         */
        private fun ByteArray.gzipCompress(): ByteArray {
            return ByteArrayOutputStream().use { baos ->
                GZIPOutputStream(baos).use { gzip ->
                    gzip.write(this)
                }
                baos.toByteArray()
            }
        }
    }
}
