package com.moloco.sdk.internal.ilrd

import com.moloco.sdk.IlrdRequest
import com.moloco.sdk.internal.ilrd.model.IlrdMediationPlatform
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.StateFlow

/**
 * Interface for Impression Level Revenue Data providers.
 * Implementers connect to various ad mediation platforms and emit
 * standardized impression events that include revenue information.
 */
interface IlrdProvider {
    /**
     * Attempts to subscribe to the mediation platform's impression events.
     * @return Result indicating success or failure of the subscription attempt
     */
    fun subscribe(): Result<Unit>

    /**
     * Identifies which mediation platform this provider integrates with
     */
    val mediationPlatform: IlrdMediationPlatform

    /**
     * Represents the current state of the provider's subscription
     * @see IlrdState
     */
    val state: StateFlow<IlrdState>

    /**
     * Flow of impression events with revenue data from the mediation platform
     */
    val events: SharedFlow<IlrdImpression>

    /**
     * Umbrella around [IlrdRequest.MaxImpression] and [IlrdRequest.LevelPlayImpression]
     * to have a common ancestor for different provider impression formats.
     */
    sealed interface IlrdImpression {
        /**
         * AppLovin MAX impression event with revenue data
         * @property impression The MAX impression object with revenue details
         */
        data class Max(val impression: IlrdRequest.MaxImpression) : IlrdImpression

        /**
         * ironSource LevelPlay impression event with revenue data
         * @property impression The LevelPlay impression object with revenue details
         */
        data class LevelPlay(val impression: IlrdRequest.LevelPlayImpression) : IlrdImpression
    }
}
