package com.moloco.sdk.internal.ilrd

import android.content.Context
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.ilrd.provider.ApplovinIlrd
import com.moloco.sdk.internal.ilrd.provider.IronsourceIlrd
import com.moloco.sdk.internal.services.bidtoken.providers.IlrdSignal
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.launch
import java.util.Collections

/**
 * Service responsible for managing Impression Level Revenue Data (ILRD) providers.
 * This class handles subscription to available ILRD providers and forwards
 * impression events to the events repository for processing and delivery.
 */
internal class IlrdService(
    private val scope: CoroutineScope,
    context: Context,
    private val eventsRepository: IlrdEventsRepository,
) {
    private val providers: List<IlrdProvider> by lazy {
        listOf(
            ApplovinIlrd(context, scope),
            IronsourceIlrd(context, scope),
        )
    }
    private val subscribedProviders = Collections.synchronizedSet(mutableSetOf<IlrdProvider>())
    private var subscribeJob: Job? = null

    fun provideDataForBidToken(): IlrdSignal? = with(eventsRepository) {
        onBidTokenRequest()

        return session?.toSignal()
            ?: run {
                MolocoLogger.warn(TAG, "provideDataForBidToken() Session is null")
                null
            }
    }
    /**
     * Subscribes to all available ILRD providers that are in Unsubscribed state.
     * This is a synchronized operation that will only execute once if already in progress.
     * Providers in Error state will be logged but not subscribed.
     */
    @Synchronized
    fun subscribe() {
        if (subscribeJob?.isActive == true) return

        subscribeJob = scope.launch {
            providers.forEach { provider ->
                provider.state.value.let { state ->
                    // In case we already have a previous state saved with error handle it
                    when (state) {
                        is IlrdState.Error -> {
                            MolocoLogger.warn(TAG, "Failed to subscribe to ${provider.mediationPlatform} ILRD: ${state.reason}")
                        }

                        is IlrdState.Unsubscribed -> {
                            subscribeToProvider(provider)
                        }

                        is IlrdState.Subscribed -> {
                            // noop
                        }
                    }
                }
            }
        }
    }

    /**
     * Attempts to subscribe to a specific ILRD provider.
     * On success, adds the provider to the active providers set and begins collecting its events.
     * On failure, logs a warning message but does not retry.
     *
     * @param provider The ILRD provider to subscribe to
     */
    private fun subscribeToProvider(provider: IlrdProvider) {
        val result = provider.subscribe()
        result.onFailure {
            MolocoLogger.warn(TAG, "Failed to subscribe to ${provider.mediationPlatform} ILRD: $it")
        }
        result.onSuccess {
            subscribedProviders.add(provider)
            provider.events.onEach {
                // Log the event
                MolocoLogger.info(TAG, "Revenue event: $it")
                eventsRepository.onEvent(it)

            }.launchIn(scope)
        }
    }

    companion object {
        private const val TAG = "IlrdService"
    }
}
/**
 * Converts an IlrdSessionManager to an IlrdSignal.
 * Creates an immutable snapshot of the current session state.
 */
private fun IlrdActiveSession.toSignal() = with(impressionCounts) {
    IlrdSignal(
        sessionId = sessionId,
        sessionStartTs = sessionStartTs,
        lastImpressionTs = lastEventReceivedTs,
        bannerImpressionCount = banner,
        mrecImpressionCount = mrec,
        nativeImpressionCount = native,
        interstitialImpressionCount = interstitial,
        rewardedImpressionCount = rewarded
    )
}
