package com.moloco.sdk.internal.ilrd.provider

import android.content.Context
import com.ironsource.mediationsdk.IronSource
import com.ironsource.mediationsdk.impressionData.ImpressionData
import com.ironsource.mediationsdk.impressionData.ImpressionDataListener
import com.moloco.sdk.IlrdRequest
import com.moloco.sdk.internal.ilrd.IlrdProvider
import com.moloco.sdk.internal.ilrd.IlrdState
import com.moloco.sdk.internal.ilrd.model.IlrdMediationPlatform
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch


/**
 * Handles impression level revenue data (ILRD) from IronSource SDK.
 *
 * IMPORTANT: This class requires the IronSource SDK to be included in the project.
 * If the IronSource SDK is not available, the subscribe and unsubscribe operations
 * will fail gracefully with log messages.
 *
 * The class uses reflection to check for IronSource's availability at runtime,
 * but cannot function without the actual SDK dependency being included by the
 * consuming application.
 */
internal class IronsourceIlrd(
    private val context: Context,
    private val scope: CoroutineScope,
) : IlrdProvider {
    override val mediationPlatform = IlrdMediationPlatform.LEVELPLAY
    override val state by lazy { _state.asStateFlow() }
    override val events by lazy { _events.asSharedFlow() }

    private val _state = MutableStateFlow<IlrdState>(IlrdState.Unsubscribed)
    private val _events = MutableSharedFlow<IlrdProvider.IlrdImpression.LevelPlay>()

    /**
     * Subscribes to IronSource impression level revenue events.
     *
     * This operation requires the IronSource SDK to be available in the project.
     * If the SDK is not available, the operation will fail gracefully with a log message.
     *
     */
    @Synchronized
    override fun subscribe(): Result<Unit> {
        val result = subscribeImpl()
        result.onFailure {
            _state.value = IlrdState.Error(it.toString())
        }
        result.onSuccess {
            _state.value = IlrdState.Subscribed
        }

        return result
    }

    /**
     * Attempts to access the IronSource SDK and subscribe to revenue events.
     *
     * This method uses reflection to check if the IronSource SDK classes are available.
     * If the IronSource SDK is not included in the project, this will return a failure result.
     *
     */
    private fun subscribeImpl(): Result<Unit> {
        runCatching {
            Class.forName("com.ironsource.mediationsdk.IronSource")
            Class.forName("com.ironsource.mediationsdk.impressionData.ImpressionData")
            Class.forName("com.ironsource.mediationsdk.impressionData.ImpressionDataListener")
        }.getOrElse { return Result.failure(it) }


        // If we are here, Ironsource is added as a dependency
        IronSource.addImpressionDataListener(createCallback())
        return Result.success(Unit)
    }

    /**
     * IMPORTANT: Lazy instance of the IronSource impression data listener. This is lazily initialized
     * to avoid instantiation if the IronSource SDK is not present.
     */
    private fun createCallback() = object : ImpressionDataListener {
        override fun onImpressionSuccess(impressionData: ImpressionData) {
            // Don't waste precious resources if we are not active
            if (scope.isActive.not()) return

            // In the case that you are subscribed to multiple topics, check for the desired one
            val model = createModel(impressionData)

            scope.launch {
                _events.emit(model)
            }
        }
    }

    private fun createModel(data: ImpressionData) =
        IlrdProvider.IlrdImpression.LevelPlay(IlrdRequest.LevelPlayImpression.newBuilder()
            .apply {
                data.auctionId?.let { setAuctionId(it) }
                data.adFormat?.let { setAdFormat(it) }
                data.adNetwork?.let { setNetworkName(it) }
                data.instanceName?.let { setInstanceName(it) }
                data.instanceId?.let { setInstanceId(it) }
                data.country?.let { setCountryCode(it) }
                data.placement?.let { setPlacement(it) }
                data.revenue?.let { setRevenue(it) }
                data.precision?.let { setPrecision(it) }
                data.ab?.let { setAb(it) }
                data.segmentName?.let { setSegmentName(it) }
                data.lifetimeRevenue?.let { setLifetimeRevenue(it) }
                data.encryptedCPM?.let { setEncryptedCpm(it) }
                data.creativeId?.let { setCreativeId(it) }
            }
            .build()
        )

    companion object {
        private const val TAG = "IronsourceIlrd"
    }
}
