package com.moloco.sdk.internal.publisher

import android.util.Base64
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.publisher.Moloco
import kotlinx.coroutines.withContext
import java.io.ByteArrayInputStream
import java.util.zip.GZIPInputStream

/**
 * Ad Load Pre-processor that based on Mediator can "transform" the input bid response from the
 * mediation SDK into a format that Moloco can process
 */
interface BidProcessor {
    /**
     * Processess the Mediation SDK's bidresponse into a bidresponse that Moloco SDK can process
     */
    suspend fun process(response: String): String?
}

internal fun createBase64GzippedBidProcessor(): BidProcessor {
    return Base64GzippedBidProcessor()
}

private const val BUFFER_SIZE = 2048
@VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
internal class Base64GzippedBidProcessor : BidProcessor {
    override suspend fun process(response: String): String? = withContext(DispatcherProvider().default) {
        MolocoLogger.debug(
            TAG,
            "Base64 Gzipped supported ${Moloco.mediationInfo?.name} mediation is detected. Starting bid response pre-process."
        )
        val processedResponse = base64DecodeThenGunzip(response)
        MolocoLogger.debug(TAG, "Processed bidresponse: $processedResponse")
        return@withContext processedResponse
    }

    private fun base64DecodeThenGunzip(response: String): String? {
        try {
            val compressed: ByteArray = Base64.decode(response, Base64.DEFAULT)

            MolocoLogger.debug(TAG, "Base64 decoded bidresponse: $compressed")
            val inputStream = ByteArrayInputStream(compressed)
            val gis = GZIPInputStream(inputStream, BUFFER_SIZE)
            val stringBuilder = StringBuilder()
            val data = ByteArray(BUFFER_SIZE)
            var bytesRead: Int
            try {
                while (gis.read(data).also { bytesRead = it } != -1) {
                    stringBuilder.append(String(data, 0, bytesRead))
                }
            } catch (e: Exception) {
                MolocoLogger.warn(TAG, "Failed to unzip bidresponse, perhaps a non-gzipped response")
                return null
            } finally {
                inputStream.close()
                gis.close()
            }
            return stringBuilder.toString()
        } catch (e: Exception) {
            MolocoLogger.warn(TAG, "Failed to base64 decode bidresponse, perhpas a non-base64 encoded response")
            return null
        }
    }

    companion object {
        private const val TAG = "Base64GzippedBidProcessor"
    }
}
