package com.moloco.sdk.internal.publisher.nativead

import android.content.Context
import android.net.Uri
import android.view.View
import android.view.ViewGroup
import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.ViewLifecycleOwner
import com.moloco.sdk.internal.publisher.nativead.model.NativeAdAssetIds
import com.moloco.sdk.internal.publisher.nativead.model.PreparedNativeAssets
import com.moloco.sdk.internal.publisher.nativead.ui.NativeAdImageContainerView
import com.moloco.sdk.internal.publisher.nativead.ui.NativeAdVideoContainer
import com.moloco.sdk.publisher.NativeAd
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ViewVisibilityTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.Ad
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.AdController
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ad.vastAdPlaylistController

internal class NativeAdAssetsProvider(
    private val context: Context,
    private val viewVisibilityTracker: ViewVisibilityTracker,
    private val viewLifecycleOwner: ViewLifecycleOwner,
    private val watermark: Watermark,
    private val vastAdPlaylistController: (Ad) -> AdController,
) : NativeAd.Assets {
    var onClick: (() -> Unit)? = null

    /**
     * Assets only get prepared once [NativeAdImpl.createAdLoadCallback] finishes processing Assets.
     */
    var preparedAssets: PreparedNativeAssets? = null

    override val title: String?
        get() = preparedAssets?.getTitlesFrom(NativeAdAssetIds.TITLE)

    override val description: String?
        get() = preparedAssets?.getDataFrom(NativeAdAssetIds.DESCRIPTION)

    override val sponsorText: String?
        get() = preparedAssets?.getDataFrom(NativeAdAssetIds.SPONSOR_TEXT)

    override val callToActionText: String?
        get() = preparedAssets?.getDataFrom(NativeAdAssetIds.CTA_TEXT)

    override val rating: Float?
        get() = preparedAssets?.getDataFrom(NativeAdAssetIds.RATING)?.toFloatOrNull()

    override val iconUri: Uri?
        get() = preparedAssets?.getImageUriFor(NativeAdAssetIds.ICON)

    override val mediaView: View?
        get() {
            // 1. Check if we have a cached videoView otherwise create it if available
            videoView.run {
                if (this != null) {
                    MolocoLogger.info(TAG, "Using cached video view")
                    removeFromParent()
                    return this
                } else {
                    preparedAssets?.getVastAdFor(NativeAdAssetIds.VIDEO)?.let { vastAd ->
                        return createVideoView(vastAd).also { videoView = it }
                    }
                }
            }

            // 2. We don't have VideoView available, so fallback to imageView.
            imageView.run {
                if (this != null) {
                    MolocoLogger.info(TAG, "Using cached image view")
                    removeFromParent()
                    return this
                } else {
                    preparedAssets?.getImageUriFor(NativeAdAssetIds.MAIN_IMAGE)?.let { imageUri ->
                        return createImageView(imageUri).also { imageView = it }
                    }
                }
            }

            // 3. No assets available. Log error
            MolocoLogger.error(TAG, "Missing video and image asset", Exception())
            return null
        }

    override val mainImageUri: Uri? = preparedAssets?.getImageUriFor(NativeAdAssetIds.MAIN_IMAGE)

    private var imageView: ViewGroup? = null
    @VisibleForTesting
    var videoView: NativeAdVideoContainer? = null

    /**
     * Creates a video view for the VAST ad, reusing a cached instance if available.
     *
     * This method checks if a video view has already been created and cached. If so, it returns the cached view.
     * Otherwise, it initializes a [vastAdPlaylistController] for handling the VAST ad playback,
     * and uses it to create a new [NativeAdVideoContainer] instance. The newly created view is cached for future use.
     *
     * @param vastAd The VAST ad object containing the data required for video playback.
     * @return A [View] instance representing the video container for the VAST ad.
     */
    private fun createVideoView(vastAd: Ad): NativeAdVideoContainer {
        val vastAdController = vastAdPlaylistController(vastAd).also {
            it.show()
        }

        return NativeAdVideoContainer(context, vastAdController, viewVisibilityTracker, viewLifecycleOwner, watermark, onClick)
    }

    /**
     * Creates an image view for the native ad, reusing a cached instance if available.
     *
     * @param imageUri The URI of the image to display in the native ad.
     * @return A [ViewGroup] instance representing the native ad image.
     */
    private fun createImageView(imageUri: Uri): ViewGroup = NativeAdImageContainerView(
        context = context,
        imageUri = imageUri,
        watermark = watermark,
        onClick = onClick
    )

    fun destroy() {
        videoView?.destroy()
        videoView = null
    }

    private fun ViewGroup.removeFromParent() {
        val parentViewGroup = this.parent as? ViewGroup
        if (parentViewGroup != null) {
            MolocoLogger.info(TAG, "Detaching view ${this.javaClass.simpleName} from parent ${parentViewGroup.javaClass.simpleName}")
            parentViewGroup.removeView(this)
        }
    }

    companion object {
        private const val TAG = "NativeAdAssetsProvider"
    }
}
