package com.moloco.sdk.internal.publisher.nativead.ui

import android.annotation.SuppressLint
import android.content.Context
import android.net.Uri
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.RelativeLayout
import com.moloco.sdk.service_locator.SdkObjectFactory
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.Watermark
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.AdBadgeView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.ButtonTracker


/**
 * A custom view that combines an image with a watermark and an ad badge.
 * The image is set via a URI, and a watermark is applied to it.
 * The ad badge is rendered at the bottom-left corner of the image view.
 *
 * @param context The context for this view.
 * @param imageUri The URI of the image to display in the image view.
 * @param watermark The watermark to apply to the image.
 * @param onClick Handle click events on the image.
 * @param externalLinkHandler The handler for external links.
 * @param buttonTracker The tracker for buttons rendered by the ad.
 */
@SuppressLint("ViewConstructor")
internal class NativeAdImageContainerView(
    context: Context,
    imageUri: Uri,
    watermark: Watermark,
    onClick: (() -> Unit)?,
    externalLinkHandler: ExternalLinkHandler = SdkObjectFactory
        .Miscellaneous.externalLinkHandlerFactory,
    buttonTracker: ButtonTracker = ButtonTracker()
) : RelativeLayout(context) {
    init {
        val imageView = ImageView(context).apply {
            setImageURI(imageUri)
            layoutParams = LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT
            )
            setOnClickListener { onClick?.invoke() }
        }

        val adBadgeView = AdBadgeView(externalLinkHandler, context)
        adBadgeView.apply {
            layoutParams = LayoutParams(
                LayoutParams.WRAP_CONTENT,
                LayoutParams.WRAP_CONTENT
            ).apply {
                addRule(ALIGN_PARENT_BOTTOM)
                addRule(ALIGN_PARENT_START)
                setPadding(AD_BADGE_PADDING, 0, 0, AD_BADGE_PADDING)
                setOnButtonRenderedListener { button ->
                    buttonTracker.onButtonRendered(button)
                }
            }
        }

        watermark.applyWatermark(imageView)
        addView(imageView)
        addView(adBadgeView)
    }

    companion object {
        /**
         * Indicates the padding at which the Ad Badge should be rendered in the bottom left corner (the i icon)
         */
        private const val AD_BADGE_PADDING = 16
    }
}
