package com.moloco.sdk.internal.services

import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.BatteryManager
import android.os.PowerManager

/**
 * Provides devices battery information to the SDK
 */
internal interface BatteryInfoService {
    /**
     * Retrieves the maximum battery level of the device
     * Possible values are 0 to 100, -1 to indicate an error
     */
    fun maxBatteryLevel(): Int

    /**
     * Retrieves whether the device is in lower power mode
     */
    fun isLowerPowerModeEnabled(): Boolean

    /**
     * Retrieves the current battery status of the device
     * Possible values (1-5), -1 to indicate error:
     * https://developer.android.com/reference/android/os/BatteryManager#BATTERY_STATUS_UNKNOWN (1)
     * https://developer.android.com/reference/android/os/BatteryManager#BATTERY_STATUS_CHARGING (2)
     * https://developer.android.com/reference/android/os/BatteryManager#BATTERY_STATUS_DISCHARGING (3)
     * https://developer.android.com/reference/android/os/BatteryManager#BATTERY_STATUS_NOT_CHARGING (4)
     * https://developer.android.com/reference/android/os/BatteryManager#BATTERY_STATUS_FULL (5)
     */
    fun batteryStatus(): Int
}


internal class BatteryInfoServiceImpl(private val context: Context) : BatteryInfoService {
    override fun maxBatteryLevel(): Int {
        val batteryStatus: Intent? = IntentFilter(Intent.ACTION_BATTERY_CHANGED).let { ifilter ->
            context.registerReceiver(null, ifilter)
        }

        return batteryStatus?.getIntExtra(BatteryManager.EXTRA_SCALE, -1) ?: -1
    }

    override fun isLowerPowerModeEnabled(): Boolean {
        val powerManager = context.getSystemService(Context.POWER_SERVICE) as PowerManager
        return powerManager.isPowerSaveMode
    }

    override fun batteryStatus(): Int {
        val batteryStatus: Intent? = IntentFilter(Intent.ACTION_BATTERY_CHANGED).let { ifilter ->
            context.registerReceiver(null, ifilter)
        }

        return batteryStatus?.getIntExtra(BatteryManager.EXTRA_STATUS, BatteryManager.BATTERY_STATUS_UNKNOWN) ?: BatteryManager.BATTERY_STATUS_UNKNOWN
    }
}