package com.moloco.sdk.internal.services

import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ExternalLinkHandler
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer.model.AdTouch
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.ButtonTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.ScreenUtils
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.flow.MutableSharedFlow

/**
 * Service responsible for handling ad click-through actions.
 */
internal interface ClickthroughService {

    /**
     * Runs the click-through process.
     *
     * @param url The URL to navigate to when the is clicked.
     * @param lastTouch The last recorded touch event on the ad.
     * @param buttonTracker Used to track button rendering for telepathy events.
     * @param clickthroughEvent An optional shared flow used to emit the click callback.
     */
    suspend fun runClickThrough(url: String,
                                lastTouch: AdTouch,
                                buttonTracker: ButtonTracker,
                                clickthroughEvent: MutableSharedFlow<Unit>? = null)


    /**
     * Runs the click-through process for a template ad. I.E the clickThrough occurred from a webview / JS layer
     * TODO: Telepathy support
     * @param url The URL to navigate to when the is clicked.
     * @param clickthroughEvent An optional shared flow used to emit the click callback.
     */
    suspend fun runTemplateAdClickThrough(url: String,
                                clickthroughEvent: MutableSharedFlow<Unit>? = null)
}

internal class ClickthroughServiceImpl(
    private val externalLinkHandler: ExternalLinkHandler,
    private val customUserEventBuilderService: CustomUserEventBuilderService
) : ClickthroughService {

    companion object{
        private const val TAG = "ClickthroughService"
    }

    override suspend fun runClickThrough(url: String,
                                         lastTouch: AdTouch,
                                         buttonTracker: ButtonTracker,
                                         clickthroughEvent: MutableSharedFlow<Unit>?) {

        val eventTimestamp = System.currentTimeMillis()
        val position = CustomUserEventBuilderService.UserInteraction.Position(
            topLeftXDp = ScreenUtils.toDp(lastTouch.touchX),
            topLeftYDp = ScreenUtils.toDp(lastTouch.touchY)
        )
        val clickEvent = CustomUserEventBuilderService.UserInteraction.Click(
            clickPosition = position,
            viewPosition = CustomUserEventBuilderService.UserInteraction.Position(
                topLeftXDp = ScreenUtils.toDp(lastTouch.viewPositionX),
                topLeftYDp = ScreenUtils.toDp(lastTouch.viewPositionY)
            ),
            viewSize = CustomUserEventBuilderService.UserInteraction.Size(
                widthDp = ScreenUtils.toDp(lastTouch.viewSizeWidth),
                heightDp = ScreenUtils.toDp(lastTouch.viewSizeHeight)
            ),
            buttonTracker.renderedButtons()
        )

        val modifiedUrl = customUserEventBuilderService
            .userAdInteractionExtAsQueryParameter(eventTimestamp, clickEvent, url)

        MolocoLogger.debug(TAG, "Launching url: $modifiedUrl")
        if (externalLinkHandler.invoke(modifiedUrl ?: "")) {
            clickthroughEvent?.emit(Unit)
        }
    }

    override suspend fun runTemplateAdClickThrough(url: String, clickthroughEvent: MutableSharedFlow<Unit>?) {
        if (externalLinkHandler.invoke(url ?: "")) {
            clickthroughEvent?.emit(Unit)
        }
    }
}
