package com.moloco.sdk.internal.services

import android.content.Context
import android.content.res.Resources
import android.hardware.Sensor
import android.hardware.SensorManager
import android.os.Build
import android.os.SystemClock
import android.telephony.TelephonyManager
import android.view.inputmethod.InputMethodManager
import android.view.inputmethod.InputMethodSubtype
import androidx.core.content.ContextCompat
import com.moloco.sdk.R
import java.util.Locale

internal interface DeviceInfoService {
    @Deprecated("Use invoke() instead", ReplaceWith("deviceInfo"))
    operator fun invoke(): DeviceInfo

    /**
     * Returns the static device information.
     */
    fun deviceInfo(): DeviceInfo

    /**
     * Returns true if the device has a gyroscope sensor.
     * null if an error occurred.
     */
    fun hasGyroscope(): Boolean?

    /**
     * Returns the device locale.
     */
    fun deviceLocale(): String

    /**
     * Returns the keyboard locale.
     */
    fun deviceKeyboardLocale(): String?
}

internal data class DeviceInfo(
    val manufacturer: String,
    val model: String,
    val hwVersion: String,
    val isTablet: Boolean,
    val os: String,
    val osVersion: String,
    val apiLevel: Int,
    val language: String,
    val mobileCarrier: String,
    val screenDensity: Float,
    val dbtMs: Long,
    val hardware: String,
    val brand: String,
)

internal class AndroidDeviceInfoService(private val context: Context) : DeviceInfoService {

    private val isTablet: Boolean by lazy {
        context.resources.getBoolean(R.bool.isTablet)
    }

    @Deprecated("Use invoke() instead", replaceWith = ReplaceWith("deviceInfo"))
    override fun invoke() = deviceInfo()

    override fun deviceInfo(): DeviceInfo = DeviceInfo(
        Build.MANUFACTURER ?: "",
        Build.MODEL ?: "",
        Build.HARDWARE ?: "",
        isTablet,
        "android",
        Build.VERSION.RELEASE,
        Build.VERSION.SDK_INT,
        Locale.getDefault().language,
        ContextCompat.getSystemService(context, TelephonyManager::class.java)
        ?.networkOperatorName ?: "",
        Resources.getSystem().displayMetrics.density,
        System.currentTimeMillis() - SystemClock.elapsedRealtime(),
        hardware = Build.HARDWARE ?: "",
        brand = Build.BRAND ?: "",
    )

    override fun hasGyroscope(): Boolean? {
        try {
            val sensorManager = context.getSystemService(Context.SENSOR_SERVICE) as SensorManager
            val gyroscopeSensor = sensorManager.getDefaultSensor(Sensor.TYPE_GYROSCOPE)
            return gyroscopeSensor != null
        } catch (e: Exception) {
            return null
        }
    }

    override fun deviceLocale(): String = Locale.getDefault().language

    override fun deviceKeyboardLocale(): String? {
        val imm = context.getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
        val currentInputMethodSubtype: InputMethodSubtype? = imm.currentInputMethodSubtype
        return currentInputMethodSubtype?.locale
    }
}
