package com.moloco.sdk.internal.services.bidtoken

import android.os.BatteryManager
import com.google.protobuf.ByteString
import com.moloco.sdk.BidToken
import com.moloco.sdk.internal.services.AdData
import com.moloco.sdk.internal.services.DeviceInfoService
import com.moloco.sdk.internal.services.MuteSwitchState
import com.moloco.sdk.internal.services.NetworkInfo
import com.moloco.sdk.internal.services.ScreenInfoService
import com.moloco.sdk.internal.services.ScreenOrientation
import com.moloco.sdk.internal.services.bidtoken.providers.ClientSignals
import com.moloco.sdk.service_locator.SdkObjectFactory
import java.util.Date
import java.util.TimeZone

/**
 * Client side token builder as per the proposal here - https://docs.google.com/document/d/1elJPg3JNxJ4s6qo1OdHDlo6mOgTNtBOHwctkJDl5r4o/edit#heading=h.rvg2y6iz8zo9
 */
internal interface ClientBidTokenBuilder {
    companion object {
        fun create(): ClientBidTokenBuilder {
            return ClientBidTokenBuilderImpl(
                SdkObjectFactory.DeviceAndApplicationInfo.deviceInfoSingleton,
                SdkObjectFactory.DeviceAndApplicationInfo.screenInfoSingleton)
        }
    }
    fun buildBidToken(bidTokenComponents: ByteArray, secret: ByteArray): ByteArray
    fun buildClientBidTokenComponent(clientSignals: ClientSignals, bidTokenConfig: BidTokenConfig): BidToken.ClientBidTokenComponents

    fun parseClientBidTokenComponent(payload: ByteArray): BidToken.ClientBidTokenComponents
    fun parse(payload: ByteArray): BidToken.ClientBidToken
}

internal class ClientBidTokenBuilderImpl(
    private val deviceInfoService: DeviceInfoService,
    private val screenInfoService: ScreenInfoService,
): ClientBidTokenBuilder {
    override fun buildBidToken(
        bidTokenComponents: ByteArray,
        secret: ByteArray,
    ): ByteArray {
        return BidToken.ClientBidToken.newBuilder().apply {
            es = ByteString.copyFrom(secret)
            payload = ByteString.copyFrom(bidTokenComponents)
        }.build().toByteArray()
    }

    override fun buildClientBidTokenComponent(
        clientSignals: ClientSignals,
        bidTokenConfig: BidTokenConfig
    ): BidToken.ClientBidTokenComponents {
        val deviceInfo = deviceInfoService.deviceInfo()
        val screenInfo = screenInfoService.screenInfo()
        return BidToken.ClientBidTokenComponents.newBuilder().apply {
            info = BidToken.ClientBidTokenComponents.SdkInfo.newBuilder().apply {
                initialized = clientSignals.sdkInitialized
            }.build()

            memoryInfo = BidToken.ClientBidTokenComponents.MemoryInfo.newBuilder().apply {
                clientSignals.memoryInfo.lowMemory?.let {
                    lowMem = it
                }
                clientSignals.memoryInfo.threshold?.let {
                    lowMemThresholdBytes = it
                }
                clientSignals.memoryInfo.totalMem?.let {
                    totalMemBytes = it
                }
            }.build()

            dirInfo = BidToken.ClientBidTokenComponents.DirInfo.newBuilder().apply {
                clientSignals.appDirInfo.appDirSize?.let {
                    dsizeBytes = it
                }
            }.build()

            networkInfo = BidToken.ClientBidTokenComponents.NetworkInfo.newBuilder().apply {
                clientSignals.networkInfoSignal.mobileCountryCode?.let {
                    mcc = clientSignals.networkInfoSignal.mobileCountryCode
                }

                clientSignals.networkInfoSignal.mobileNetworkCode?.let {
                    mnc = it
                }
                clientSignals.networkInfoSignal.networkRestricted?.let {
                    restricted = it
                }

                clientSignals.networkInfoSignal.networkType?.let {
                    type = when (it) {
                        is NetworkInfo.CellularNetwork -> BidToken.ClientBidTokenComponents.NetworkInfo.ConnectionType.CELLULAR
                        NetworkInfo.NoNetwork -> BidToken.ClientBidTokenComponents.NetworkInfo.ConnectionType.NO_NETWORK
                        NetworkInfo.WifiNetwork -> BidToken.ClientBidTokenComponents.NetworkInfo.ConnectionType.WIFI
                    }
                }

            }.build()

            batteryInfo = BidToken.ClientBidTokenComponents.BatteryInfo.newBuilder().apply {
                clientSignals.batteryInfoSignal.maxBatteryLevel?.let {
                    maxBatteryLevel = it
                }

                clientSignals.batteryInfoSignal.batteryStatus?.let {
                    batteryStatus = when (it) {
                        BatteryManager.BATTERY_STATUS_CHARGING -> BidToken.ClientBidTokenComponents.BatteryInfo.BatteryStatus.CHARGING
                        BatteryManager.BATTERY_STATUS_DISCHARGING -> BidToken.ClientBidTokenComponents.BatteryInfo.BatteryStatus.DISCHARGING
                        BatteryManager.BATTERY_STATUS_NOT_CHARGING -> BidToken.ClientBidTokenComponents.BatteryInfo.BatteryStatus.NOT_CHARGING
                        BatteryManager.BATTERY_STATUS_FULL -> BidToken.ClientBidTokenComponents.BatteryInfo.BatteryStatus.FULL
                        else -> BidToken.ClientBidTokenComponents.BatteryInfo.BatteryStatus.UNKNOWN
                    }
                }

                clientSignals.batteryInfoSignal.isPowerSaveMode?.let {
                    lowPowMode = it
                }
            }.build()

            adInfo = BidToken.ClientBidTokenComponents.AdvertisingInfo.newBuilder().apply {
                when (val adData = clientSignals.adDataSignal) {
                    is AdData.Available -> {
                        dnt = false
                        id = adData.id
                    }
                    AdData.Unavailable -> {
                        dnt = true
                    }
                }
            }.build()

            privacy = BidToken.ClientBidTokenComponents.Privacy.newBuilder().apply {
                clientSignals.privacySettings.isAgeRestrictedUser?.let {
                    coppa = it
                }
                clientSignals.privacySettings.isUserConsent?.let {
                    gdpr = it
                }
                clientSignals.privacySettings.isDoNotSell?.let {
                    ccpa = it
                }
                clientSignals.privacySettings.TCFConsent?.let {
                    tcfConsentString = it
                }
                usPrivacy = clientSignals.privacySettings.usPrivacy
            }.build()

            device = BidToken.ClientBidTokenComponents.Device.newBuilder().apply {
                // 1
                language = deviceInfo.language
                // 2
                osv = deviceInfo.osVersion
                // 3
                make = deviceInfo.manufacturer
                // 4
                model = deviceInfo.model
                // 5
                hwv = deviceInfo.hwVersion
                // 6
                carrier = deviceInfo.mobileCarrier
                // 7
                devicetype = if (deviceInfo.isTablet) 5 else 1
                // 8
                js = 1
                // 9
                geo = BidToken.ClientBidTokenComponents.Geo.newBuilder().apply {
                    utcoffset = TimeZone.getDefault().getOffset(Date().time) / 60000
                }.build()
                // 10
                w = screenInfo.screenWidthPx
                // 11
                h = screenInfo.screenHeightPx
                // 12
                pxratio = screenInfo.density.toDouble()
                // 13
                ppi = screenInfo.dpi
                // 14
                os = deviceInfo.os
                // 15
                if (bidTokenConfig.dbtEnabled) {
                    dbt = deviceInfo.dbtMs.millisToNanos()
                }
                // 16
                clientSignals.deviceSignal.orientation?.let {
                    ortn = it.toProtoOrientation()
                }
                // 17
                deviceInfoService.hasGyroscope()?.let {
                    hasGy = it
                }
                // 18
                clientSignals.deviceSignal.keyboardLocale?.let {
                    kbLoc = it
                }

                // 19
                clientSignals.deviceSignal.locale?.let {
                    locale = it
                }

                // 20
                xdpi = screenInfo.xdpi
                // 21
                ydpi = screenInfo.ydpi

                // 22
                hardware = deviceInfo.hardware
                // 23
                brand = deviceInfo.brand
            }.build()

            audioInfo = BidToken.ClientBidTokenComponents.AudioInfo.newBuilder().apply {
                clientSignals.audioSignal.muteSwitchState?.let {
                    muteSwitch = it.toProtoMuteSwitchState()
                }

                clientSignals.audioSignal.mediaVolume?.let {
                    vol = it
                }
            }.build()

            accessibilityInfo = BidToken.ClientBidTokenComponents.AccessibilityInfo.newBuilder().apply {
                clientSignals.accessibilitySignal.fontScale?.let {
                    fontScale = it
                }
                clientSignals.accessibilitySignal.accessibilityLargePointerIcon?.let {
                    accessibilityLargePointerIcon = it
                }
                clientSignals.accessibilitySignal.accessibilityCaptioningEnabled?.let {
                    accessibilityCaptioningEnabled = it
                }
                clientSignals.accessibilitySignal.reduceBrightColorsActivated?.let {
                    reduceBrightColorsActivated = it
                }
            }.build()

            if (clientSignals.ilrdSignal.isEnabled) {
                impLvlRevData = BidToken.ClientBidTokenComponents.ImpLvlRevData.newBuilder().apply {
                    sessionId = clientSignals.ilrdSignal.sessionId
                    lastImpTs = clientSignals.ilrdSignal.lastImpressionTs
                    sessionStartTs = clientSignals.ilrdSignal.sessionStartTs
                    impCounts = BidToken.ClientBidTokenComponents.ImpLvlRevData.ImpCounts.newBuilder().apply {
                        banner = clientSignals.ilrdSignal.bannerImpressionCount
                        mrec = clientSignals.ilrdSignal.mrecImpressionCount
                        native = clientSignals.ilrdSignal.nativeImpressionCount
                        interstitial = clientSignals.ilrdSignal.interstitialImpressionCount
                        native = clientSignals.ilrdSignal.nativeImpressionCount
                    }.build()
                }.build()
            }
        }.build()
    }

    override fun parse(payload: ByteArray): BidToken.ClientBidToken {
        return BidToken.ClientBidToken.parseFrom(payload)
    }

    override fun parseClientBidTokenComponent(payload: ByteArray): BidToken.ClientBidTokenComponents {
        return BidToken.ClientBidTokenComponents.parseFrom(payload)
    }

    private fun Long.millisToNanos(): Long = this * 1_000_000

    private fun ScreenOrientation.toProtoOrientation(): BidToken.ClientBidTokenComponents.Device.Orientation =
        when (this) {
            ScreenOrientation.UNKNOWN -> BidToken.ClientBidTokenComponents.Device.Orientation.UNKNOWN
            ScreenOrientation.PORTRAIT -> BidToken.ClientBidTokenComponents.Device.Orientation.PORTRAIT
            ScreenOrientation.LANDSCAPE -> BidToken.ClientBidTokenComponents.Device.Orientation.LANDSCAPE
        }

    private fun MuteSwitchState.toProtoMuteSwitchState(): BidToken.ClientBidTokenComponents.AudioInfo.MuteSwitchState =
        when (this) {
            MuteSwitchState.SILENT -> BidToken.ClientBidTokenComponents.AudioInfo.MuteSwitchState.SILENT
            MuteSwitchState.VIBRATE -> BidToken.ClientBidTokenComponents.AudioInfo.MuteSwitchState.VIBRATE
            MuteSwitchState.NORMAL -> BidToken.ClientBidTokenComponents.AudioInfo.MuteSwitchState.NORMAL
        }
}
