package com.moloco.sdk.internal.services.bidtoken

import androidx.annotation.VisibleForTesting
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.bidtoken.BidToken
import com.moloco.sdk.internal.bidtoken.BidTokenParser
import com.moloco.sdk.internal.services.TimeProviderService
import com.moloco.sdk.internal.services.bidtoken.ServerBidTokenCacheImpl.Companion.EXPIRY_THRESHOLD_MINS
import java.util.concurrent.TimeUnit

internal enum class ServerTokenCacheStatus {
    NEEDS_REFRESH,
    EXPIRING,
    NO_REFRESH_NEEDED;

    internal fun canUseToken() = this == EXPIRING || this == NO_REFRESH_NEEDED
}

internal interface ServerBidTokenCache {
    val cachedToken: BidTokenResponseComponents

    /**
     * @return The status of the token cache.
     */
    suspend fun tokenStatus(): ServerTokenCacheStatus

    /**
     * Updates the cache with the new bid token if the new token is valid and newer.
     * @param bidTokenComponents The new bid token components.
     */
    suspend fun updateCache(bidTokenComponents: BidTokenResponseComponents)

    /**
     * Clears the cache.
     */
    fun clearCache()

    companion object {
        fun create(bidTokenParser: BidTokenParser, timeProviderService: TimeProviderService): ServerBidTokenCache {
            return ServerBidTokenCacheImpl(bidTokenParser, timeProviderService)
        }
    }
}

internal class ServerBidTokenCacheImpl(
    private val bidTokenParser: BidTokenParser,
    private val timeProviderService: TimeProviderService,
) : ServerBidTokenCache {
    @VisibleForTesting
    internal var _cachedToken = BidTokenResponseComponents("", "", DefaultBidTokenConfig)

    override val cachedToken: BidTokenResponseComponents
        get() = _cachedToken

    override suspend fun tokenStatus(): ServerTokenCacheStatus {
        if (cachedToken.bidToken.isEmpty()) {
            MolocoLogger.info(TAG, "[Thread: ${Thread.currentThread().name}] cached bidToken is empty, needs refresh")
            return ServerTokenCacheStatus.NEEDS_REFRESH
        }

        when(val bidTokenParseResult = bidTokenParser(cachedToken.bidToken)) {
            is Result.Failure -> {
                MolocoLogger.info(TAG, "[Thread: ${Thread.currentThread().name}] Failed to parse cached token for expiration, needs refresh")
                return ServerTokenCacheStatus.NEEDS_REFRESH
            }
            is Result.Success -> {
                val bidToken = bidTokenParseResult.value
                val currentTimeInMillis = timeProviderService.currentTime()
                if (bidToken.isExpired(currentTimeInMillis)) {
                    MolocoLogger.info(TAG, "[Thread: ${Thread.currentThread().name}] Bid token expired, needs refresh")
                    return ServerTokenCacheStatus.NEEDS_REFRESH
                } else if (bidToken.isNearExpiry(currentTimeInMillis)) {
                    MolocoLogger.info(TAG, "[Thread: ${Thread.currentThread().name}] Bid token is near expiry. It will expire soon")
                    return ServerTokenCacheStatus.EXPIRING
                } else {
                    MolocoLogger.info(TAG, "[Thread: ${Thread.currentThread().name}] Bid token has not expired")
                }
            }
        }

        MolocoLogger.info(TAG, "[Thread: ${Thread.currentThread().name}] Bid token doesn't need refresh")
        return ServerTokenCacheStatus.NO_REFRESH_NEEDED
    }

    override suspend fun updateCache(bidTokenComponents: BidTokenResponseComponents) {
        debugBuildLog("[Thread: ${Thread.currentThread().name}] Acquired lock, checking for new token expiry")

        if (_cachedToken.bidToken == "") {
            _cachedToken = bidTokenComponents
            debugBuildLog("[Thread: ${Thread.currentThread().name}] Updated cache with new bidToken as existing token was empty")
            return
        }

        val newTokenParseResult = bidTokenParser(bidTokenComponents.bidToken)
        if (newTokenParseResult is Result.Success) {
            val newToken = newTokenParseResult.value
            val currentTokenParseResult = bidTokenParser(_cachedToken.bidToken)
            if (currentTokenParseResult is Result.Success) {
                val currentToken = currentTokenParseResult.value
                if (newToken.expiresAtUnixSeconds > currentToken.expiresAtUnixSeconds) {
                    _cachedToken = bidTokenComponents
                    debugBuildLog("[Thread: ${Thread.currentThread().name}] Updated cache with new bidToken")
                } else {
                    debugBuildLog("[Thread: ${Thread.currentThread().name}] New token's expiration is not greater than the existing token's expiration. Cache not updated.")
                }
            } else {
                // If current token parsing fails, update the cache with the new token
                _cachedToken = bidTokenComponents
                debugBuildLog("[Thread: ${Thread.currentThread().name}] Current token parsing failed. Updated cache with new bidToken")
            }
        } else {
            debugBuildLog("[Thread: ${Thread.currentThread().name}] New token parsing failed. Cache not updated.")
        }
    }

    override fun clearCache() {
        _cachedToken = BidTokenResponseComponents("", "", DefaultBidTokenConfig)
    }

    private fun debugBuildLog(message: String) {
        MolocoLogger.debugBuildLog(TAG, "[Thread: ${Thread.currentThread().name}][sbt] $message")
    }

    companion object {
        private const val TAG = "ServerBidTokenCache"
        internal const val EXPIRY_THRESHOLD_MINS = 15L // BidToken will be considered near expiry 15 minutes before the server token expiry.
        internal const val EXPIRED_THRESHOLD_MINS = 2L // BidToken will considered expired 2 minutes before the server token expiry
    }
}

/**
 * @return True if the bid token is near expiry, false otherwise.
 * A bid token is considered near expiry if it will expire in the next [EXPIRY_THRESHOLD_MINS] minutes.
 * A bid token is considered not near expiry if it has already expired (i.e will return false)
 */
@VisibleForTesting
internal fun BidToken.isNearExpiry(currentTimeInMillis: Long): Boolean {
    val expiryTimeMillis = TimeUnit.SECONDS.toMillis(this.expiresAtUnixSeconds)
    val nearExpiryThresholdMillis = TimeUnit.MINUTES.toMillis(EXPIRY_THRESHOLD_MINS)
    val expiring = !isExpired(currentTimeInMillis) && expiryTimeMillis - currentTimeInMillis <= nearExpiryThresholdMillis
    MolocoLogger.debugBuildLog("ServerBidTokenCache", "[sbt] currentTimeInMillis: $currentTimeInMillis, expiryTimeMillis: $expiryTimeMillis, nearExpiryThresholdMillis: $nearExpiryThresholdMillis, expiring: $expiring")
    return expiring
}

@VisibleForTesting
/**
 * @return True if the bid token has expired, false otherwise.
 * A bid token is considered expired if it will expire
 * within [EXPIRED_THRESHOLD_MINS] minutes of the token expiry
 * or current time is already past expiration time
 *
 */
internal fun BidToken.isExpired(currentTimeInMillis: Long): Boolean {
    val expirationTimeMillis = TimeUnit.SECONDS.toMillis(this.expiresAtUnixSeconds)
    val expiredThresholdMillis = TimeUnit.MINUTES.toMillis(ServerBidTokenCacheImpl.EXPIRED_THRESHOLD_MINS)
    val expired = currentTimeInMillis >= (expirationTimeMillis - expiredThresholdMillis)
    MolocoLogger.debugBuildLog("ServerBidTokenCache", "[sbt] currentTimeInMillis: $currentTimeInMillis, expirationTimeMillis: $expirationTimeMillis, expiredThresholdMillis: $expiredThresholdMillis, expired: $expired")
    return expired
}

