package com.moloco.sdk.internal.services.bidtoken.providers

import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.ilrd.IlrdService

internal data class IlrdSignal(
    /**
     * By convention, if this is blank, then the ILRD is disabled.
     */
    val sessionId: String,
    val sessionStartTs: Long,
    val lastImpressionTs: Long,
    val bannerImpressionCount: Int,
    val mrecImpressionCount: Int,
    val nativeImpressionCount: Int,
    val interstitialImpressionCount: Int,
    val rewardedImpressionCount: Int,
){
    val isEnabled get() = sessionId.isNotEmpty()
}

internal class IlrdSignalProvider(
    private val _ilrdService: () -> IlrdService?,
) : ClientBidTokenSignalProvider<IlrdSignal> {
    private var cachedSignal = getIlrdSignal()

    override fun needsRefresh(): Boolean {
        val currentSignal = getIlrdSignal()
        val needsRefresh = currentSignal != cachedSignal
        MolocoLogger.debugBuildLog(TAG, "[Ilrd] needsRefresh: $needsRefresh, with current: $currentSignal, cached: $cachedSignal")
        return needsRefresh
    }

    override fun tryUpdateSignalState() {
        cachedSignal = getIlrdSignal()
    }

    override fun provideSignal(): IlrdSignal {
        return cachedSignal
    }

    override fun logTag(): String = TAG

    /**
     * Retrieves the current ILRD signal from the service.
     *
     * If the service is unavailable, returns a default disabled signal.
     *
     * @return The current ILRD signal or a default disabled signal
     */
    private fun getIlrdSignal() = try {
        _ilrdService()?.provideDataForBidToken() ?: createDisabledIlrdSignal()
    } catch (e: Exception) {
        MolocoLogger.error(TAG, "Error retrieving ILRD signal", e)
        createDisabledIlrdSignal()
    }

    private fun createDisabledIlrdSignal() = IlrdSignal(
        sessionId = "",
        sessionStartTs = -1,
        lastImpressionTs = -1,
        bannerImpressionCount = -1,
        mrecImpressionCount = -1,
        nativeImpressionCount = -1,
        interstitialImpressionCount = -1,
        rewardedImpressionCount = -1,
    )

    companion object {
        private const val TAG = "IlrdSignalProvider"
    }
}
